<?php
session_start();
include('../inc/dbcon.php');
include('../inc/functions.php');
include('../inc/cartfunctions.php');
include('../inc/repfunctions.php');
include('../inc/invfunctions.php');

if (!isset($_SESSION["axeslogin"]) || $_SESSION["axeslogin"] !== true) {
    header('Location:../index.php');
    exit;
}

$aid = $_SESSION['uid'];
$uty = $_SESSION['utype'];
$brid = $_SESSION['abrid'];
$today = strftime("%Y-%m-%d", time());
$dtnow = date("Y-m-d h:i:s", time());

/**
 * Checks if a purchase invoice can be deleted.
 * Optimized to use a single query for related products and pre-fetch other checks.
 *
 * @param string $invno The invoice number to check.
 * @param mysqli $con The database connection.
 * @return int 1 if cannot be deleted, 0 otherwise.
 */
function check_purinvdel($invno, $con)
{
    // Check for related products with specific modifications (SE, TR, WR)
    $stmt = $con->prepare("
        SELECT 1
        FROM tbl_traproduct tp
        WHERE tp.invno = ?
        AND EXISTS (
            SELECT 1
            FROM tbl_traproduct sub_tp
            WHERE sub_tp.unqid = tp.unqid
            AND sub_tp.mods IN ('SE', 'TR', 'WR')
        )
        LIMIT 1
    ");
    $stmt->bind_param("s", $invno);
    $stmt->execute();
    $stmt->store_result();
    if ($stmt->num_rows > 0) {
        $stmt->close();
        return 1;
    }
    $stmt->close();

    // Check for references in tbl_trarecord
    $stmt = $con->prepare("SELECT 1 FROM tbl_trarecord WHERE invno != ? AND refinv = ? LIMIT 1");
    $stmt->bind_param("ss", $invno, $invno);
    $stmt->execute();
    $stmt->store_result();
    if ($stmt->num_rows > 0) {
        $stmt->close();
        return 1;
    }
    $stmt->close();

    // Check for references in tbl_preturn
    $stmt = $con->prepare("SELECT 1 FROM tbl_preturn WHERE refinv = ? LIMIT 1");
    $stmt->bind_param("s", $invno);
    $stmt->execute();
    $stmt->store_result();
    if ($stmt->num_rows > 0) {
        $stmt->close();
        return 1;
    }
    $stmt->close();

    return 0;
}

$request = $_REQUEST;

// --- Build Base Query ---
$base_query = "FROM tbl_purchase";
$where_clauses = [];
$params = [];
$param_types = '';

if ($uty != '1') {
    $where_clauses[] = "brid = ?";
    $params[] = $brid;
    $param_types .= 'i';
}

// --- Search Logic ---
$search_value = $request['search']['value'];
$searchable_columns = ['apdate', 'invno', 'total', 'ref','name']; // Explicitly define searchable columns
$search_conditions = [];

if (!empty($search_value)) {
    foreach ($searchable_columns as $column) {
        $search_conditions[] = "`" . $column . "` LIKE ?";
        $params[] = '%' . $search_value . '%';
        $param_types .= 's';
    }
    $where_clauses[] = '(' . implode(' OR ', $search_conditions) . ')';
}

$sql_count = "SELECT COUNT(id) AS total_rows " . $base_query;
if (!empty($where_clauses)) {
    $sql_count .= " WHERE " . implode(' AND ', $where_clauses);
}

$stmt_count = $con->prepare($sql_count);
if (!empty($params)) {
    $stmt_count->bind_param($param_types, ...$params);
}
$stmt_count->execute();
$result_count = $stmt_count->get_result();
$totalData = $result_count->fetch_assoc()['total_rows'];
$totalFilter = $totalData; // For this specific search, filtered count is the same as total data count if no other filters applied

$stmt_count->close();

// --- Main Data Query ---
$sql_data = "SELECT id, apdate, brid,date,note,nxtduedate, type, supid, invno, total, ref, refinv " . $base_query;
if (!empty($where_clauses)) {
    $sql_data .= " WHERE " . implode(' AND ', $where_clauses);
}
$sql_data .= " ORDER BY apdate DESC, id DESC LIMIT ?, ?";

$params_data = $params; // Copy params for data query
$param_types_data = $param_types; // Copy param types for data query

// Add limit parameters
$params_data[] = (int)$request['start'];
$params_data[] = (int)$request['length'];
$param_types_data .= 'ii';

$stmt_data = $con->prepare($sql_data);
if (!empty($params_data)) {
    $stmt_data->bind_param($param_types_data, ...$params_data);
}
$stmt_data->execute();
$query = $stmt_data->get_result();

$data = [];
$s = 0;

// Pre-fetch all necessary related data to avoid N+1 queries
$branch_ids = [];
$supplier_ids = [];
$customer_ids = [];
$invoice_numbers = [];

while ($row_temp = $query->fetch_assoc()) {
    if ($uty == '1' && $row_temp['brid'] != 0) {
        $branch_ids[] = $row_temp['brid'];
    }
    if ($row_temp['type'] == 'SU') {
        $supplier_ids[] = $row_temp['supid'];
    } else {
        $customer_ids[] = $row_temp['supid'];
    }
    $invoice_numbers[] = $row_temp['invno'];
}
$query->data_seek(0); // Reset result pointer

// Fetch branch names
$branch_names = [];
if (!empty($branch_ids)) {
    $branch_ids_str = implode(',', array_unique($branch_ids));
    $res_branch = mysqli_query($con, "SELECT id, name FROM tbl_branch WHERE id IN ($branch_ids_str)");
    while ($b_row = mysqli_fetch_assoc($res_branch)) {
        $branch_names[$b_row['id']] = $b_row['name'];
    }
}

// Fetch supplier data
$supplier_data = [];
if (!empty($supplier_ids)) {
    $supplier_ids_str = implode(',', array_unique($supplier_ids));
    $res_supplier = mysqli_query($con, "SELECT id, name, cnumber, cemail, code FROM tbl_supplier WHERE id IN ($supplier_ids_str)");
    while ($s_row = mysqli_fetch_assoc($res_supplier)) {
        $supplier_data[$s_row['id']] = $s_row;
    }
}

// Fetch customer data
$customer_data = [];
if (!empty($customer_ids)) {
    $customer_ids_str = implode(',', array_unique($customer_ids));
    $res_customer = mysqli_query($con, "SELECT id, name, cnumber, cemail, code FROM tbl_customer WHERE id IN ($customer_ids_str)");
    while ($c_row = mysqli_fetch_assoc($res_customer)) {
        $customer_data[$c_row['id']] = $c_row;
    }
}

// Check deletability for all invoices in one go (or in batches if too many)
// This could still be a performance bottleneck if invoice_numbers is very large.
// Consider caching the results of check_purinvdel or further optimizing it.
$invoice_deletability = [];
foreach (array_unique($invoice_numbers) as $inv_num) {
    $invoice_deletability[$inv_num] = check_purinvdel($inv_num, $con);
}


// Get currency symbol once
$currency_symbol = get_fild_data('tbl_currency', '1', 'symbol');


while ($row = $query->fetch_assoc()) {
    $ac = '';
    $s += 1;
    $invno = $row['invno'];
    $subdata = [];
    $subdata[] = $s + $request['start']; // serial
    $subdata[] = date("d M Y", strtotime($row['apdate'])); // date

    if ($uty == '1') {
        if ($row['brid'] == 0) {
            $subdata[] = 'Main Branch';
        } else {
            $subdata[] = $branch_names[$row['brid']] ?? '';
        }
    }

    if ($row['type'] == 'SU') {
        $sup_info = $supplier_data[$row['supid']] ?? [];
        $subdata[] = show_addres(
            $sup_info['name'] ?? '',
            $sup_info['cnumber'] ?? '',
            $sup_info['cemail'] ?? '',
            '',
            $sup_info['code'] ?? '',
            $row['type'] . '_' . $row['supid']
        );
    } else {
        $cust_info = $customer_data[$row['supid']] ?? [];
        $subdata[] = show_addres(
            $cust_info['name'] ?? '',
            $cust_info['cnumber'] ?? '',
            $cust_info['cemail'] ?? '',
            '',
            $cust_info['code'] ?? '',
            $row['type'] . '_' . $row['supid']
        );
    }

    $subdata[] = $row['invno'];
    $subdata[] = numtolocal($row['total'], $currency_symbol);
    //$subdata[] = $row['ref'];
    
    
     if($row['nxtduedate']!=''){
        $subdata[]=$row['ref'].' '.$row['note'].'<br>Next Due Date: '.date("d M Y", strtotime($row['nxtduedate'])).'<br><strong>Created By: </strong>'.get_fild_data('tbl_user',$row['uid'],'name').' '.date("d M Y H:i:s", strtotime($row['date'])); 
    }else{
           $subdata[]=$row['ref'].' '.$row['note'].'<br><strong>Created By: </strong>'.get_fild_data('tbl_user',$row['uid'],'name').' '.date("d M Y H:i:s", strtotime($row['date']));
    }

    $can_delete = $invoice_deletability[$invno] ?? 1; // Default to not deletable if not found

    if (get_pagesecurity('48', 'prints', 'P')) {
        $ac .= '<a class="btn btn-flat bg-purple details-invoice" style="margin: 3px;" data-toggle="tooltip" title="View" href="#" id="inv_' . $row['id'] . '_' . $row['supid'] . '"><i class="fa fa-eye cat-child"></i></a>';
    }

    if (!$can_delete) {
        if (substr($row['refinv'], 0, 3) == 'POR') {
            // Do nothing if refinv starts with POR and cannot be deleted
        } else {
            if (get_pagesecurity('48', 'edits', 'P')) {
                $ac .= '<a class="btn btn-flat bg-purple" style="margin: 3px;" data-toggle="tooltip" title="Edit" href="#" onclick="take_action(\'ED_' . $row['id'] . '\')"><i class="fa fa-edit"></i></a>';
            }
        }
    }

    $ac .= '<a class="btn btn-flat bg-purple" style="margin: 3px;" href="#" data-toggle="tooltip" title="Return" onclick="take_action(\'RE_' . $row['id'] . '\')"><i class="fa fa-reply"></i></a>';

    if (!$can_delete) {
        if (get_pagesecurity('48', 'deletes', 'P')) {
            $ac .= '<a class="btn btn-flat bg-purple" style="margin: 3px;" href="#" data-toggle="tooltip" title="Delete" onclick="remove_item(\'DL_' . $row['id'] . '\')"><i class="fa fa-trash"></i></a>';
        }
    }

    if (!$can_delete) {
        $ac .= '<form action="pur_pinvedit.php" id="ED_' . $row['id'] . '" method="post" >';
        $ac .= '<input type="hidden" name="pured" value="' . $row['invno'] . '" />';
        $ac .= '</form>';
    }

    $ac .= '<form action="pur_preturncteate.php" id="RE_' . $row['id'] . '" method="post" >';
    $ac .= '<input type="hidden" name="puret" value="' . $row['id'] . '" />';
    $ac .= '</form>';
    $ac .= '<form action="pur_pinvlist.php" id="DL_' . $row['id'] . '" method="post" >';
    $ac .= '<input type="hidden" name="delpur" value="' . $row['id'] . '" />';
    $ac .= '</form>';

    $subdata[] = $ac;
    $data[] = $subdata;
}

$stmt_data->close();

$json_data = [
    "draw" => intval($request['draw']),
    "recordsTotal" => intval($totalData),
    "recordsFiltered" => intval($totalFilter),
    "data" => $data
];

echo json_encode($json_data);
?>