@extends("general_base")
@section("title", "Order Invoice")

@section("style")
<style>
.success-message {
    background: #e8f5e9;
    color: #1b5e20;
    padding: 1.2rem;
    border-radius: 8px;
    margin-bottom: 2rem;
    text-align: center;
    font-size: 1.1rem;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 10px;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
}

.success-message i {
    font-size: 1.5rem;
    color: #2e7d32;
}

.invoice-container {
    background: #fff;
    border-radius: 12px;
    box-shadow: 0 2px 20px rgba(0, 0, 0, 0.06);
    margin-bottom: 2rem;
    position: relative;
    overflow: hidden;
}

.invoice-header {
    padding: 2.5rem 2rem 2rem;
}

.invoice-top {
    padding: 2rem 0;
}

.company-logo {
    text-align: left;
}

.company-logo img {
    height: 45px;
}

.invoice-title {
    color: #1a202c;
    font-size: 1.75rem;
    font-weight: 600;
    margin-bottom: 2rem;
    padding-bottom: 1rem;
    border-bottom: 1px solid #e2e8f0;
}

.invoice-title h2 {
    color: #1a202c;
    font-size: 1.5rem;
    font-weight: 600;
    margin: 0;
}

.invoice-title .order-id {
    color: #4a5568;
    font-size: 1.1rem;
    margin-top: 0.5rem;
    font-weight: 500;
}

.invoice-meta {
    display: flex;
    justify-content: space-between;
    padding: 1.5rem 0;
    border-bottom: 1px solid #e2e8f0;
}

.meta-group {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.meta-label {
    color: #6b7280;
    font-size: 0.875rem;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.meta-value {
    color: #1a202c;
    font-size: 1rem;
    font-weight: 500;
}

.meta-item {
    display: flex;
    gap: 0.5rem;
    font-size: 0.95rem;
    line-height: 1.4;
}

.meta-item span:first-child {
    color: #64748b;
    font-size: 0.9rem;
}

.meta-item span:last-child {
    color: #1a202c;
    font-weight: 500;
}

.invoice-header::after {
    content: '';
    position: absolute;
    bottom: -1px;
    left: 0;
    right: 0;
    height: 1px;
    background: linear-gradient(90deg, transparent, #e2e8f0, transparent);
}

.invoice-header h2 {
    color: #1a202c;
    margin-bottom: 1rem;
    font-size: 1.5rem;
    font-weight: 600;
}

.invoice-meta-item {
    display: flex;
    flex-direction: column;
    gap: 0.3rem;
}

.invoice-meta-item strong {
    color: #2d3748;
    font-size: 1rem;
}

.invoice-content {
    padding: 2rem;
}

.invoice-section {
    margin-bottom: 2.5rem;
}

.invoice-section h3 {
    color: #2d3748;
    font-size: 1.2rem;
    font-weight: 600;
    margin-bottom: 1.2rem;
    padding-bottom: 0.8rem;
    border-bottom: 2px solid #edf2f7;
    position: relative;
}

.company-info {
    text-align: center;
    padding-bottom: 1.5rem;
    margin-bottom: 1.5rem;
    border-bottom: 1px solid #edf2f7;
}

.company-info img {
    height: 50px;
    margin-bottom: 0.5rem;
}

.company-info p {
    color: #4a5568;
    font-size: 0.9rem;
    margin: 0;
}

.shipping-grid {
    display: grid;
    grid-template-columns: repeat(2, 1fr);
    gap: 1.5rem;
}

.shipping-box {
    background: #f8fafc;
    padding: 1.2rem;
    border-radius: 8px;
    border: 1px solid #edf2f7;
}

.shipping-box h4 {
    font-size: 0.85rem;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    color: #64748b;
    margin-bottom: 1rem;
    padding-bottom: 0.5rem;
    border-bottom: 1px solid #edf2f7;
}

.shipping-info {
    display: grid;
    gap: 0.5rem;
}

.shipping-info p {
    margin: 0;
    color: #4a5568;
    display: flex;
    justify-content: space-between;
    font-size: 0.95rem;
}

.shipping-info p span:first-child {
    color: #64748b;
    font-size: 0.9rem;
}

.shipping-info p span:last-child {
    color: #1a202c;
    font-weight: 500;
}

.shipping-details {
    background: #f8fafc;
    padding: 1.5rem;
    border-radius: 8px;
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
}

.shipping-details p {
    margin: 0;
    color: #4a5568;
    font-size: 0.95rem;
}

.shipping-details strong {
    color: #2d3748;
    display: block;
    margin-bottom: 0.3rem;
    font-size: 0.9rem;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.order-items {
    border: 1px solid #edf2f7;
    border-radius: 8px;
    overflow: hidden;
}

.order-items th {
    background: #f8fafc;
    padding: 1rem;
    text-align: left;
    color: #2d3748;
    font-weight: 600;
    font-size: 0.9rem;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    border-bottom: 2px solid #edf2f7;
}

.order-items td {
    padding: 1rem;
    border-bottom: 1px solid #edf2f7;
    color: #4a5568;
    vertical-align: middle;
}

.order-items tr:last-child td {
    border-bottom: none;
}

.item-image {
    width: 70px;
    height: 70px;
    object-fit: cover;
    border-radius: 8px;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
}

.item-details {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.item-info strong {
    color: #2d3748;
    font-weight: 500;
    display: block;
    margin-bottom: 0.3rem;
}

.item-info span {
    color: #718096;
    font-size: 0.9rem;
}

.order-summary {
    margin-top: 1.5rem;
    background: #f8fafc;
    padding: 1rem 1.5rem;
    border-radius: 8px;
}

.summary-row {
    display: flex;
    justify-content: space-between;
    padding: 0.5rem 0;
    color: #4a5568;
    font-size: 0.95rem;
}

.summary-row:not(:last-child) {
    border-bottom: 1px dashed #edf2f7;
}

.summary-row.discount {
    color: #dc2626;
}

.summary-row.total {
    font-size: 1.2rem;
    font-weight: 600;
    color: #2d3748;
    border-top: 2px solid #e2e8f0;
    margin-top: 0.5rem;
    padding-top: 0.8rem;
    border-bottom: none;
}

.payment-info {
    background: #fffbeb;
    color: #92400e;
    padding: 1rem 1.5rem;
    border-radius: 8px;
    margin-top: 1.5rem;
    display: flex;
    align-items: center;
    gap: 0.8rem;
    font-size: 0.95rem;
}

.payment-info i {
    font-size: 1.2rem;
}

.action-buttons {
    margin-top: 2rem;
    display: flex;
    justify-content: center;
    gap: 1rem;
    flex-wrap: wrap;
}

.action-buttons .mn-btn-2 {
    padding: 0.8rem 1.5rem;
    min-width: 200px;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
    font-weight: 500;
}

.action-buttons .mn-btn-2 i {
    font-size: 1.2rem;
}

.product-group {
    background: #fff;
}

.product-group:not(:last-child) {
    border-bottom: 1px solid #edf2f7;
}

.product-group-header {
    display: flex;
    align-items: center;
    gap: 1rem;
    padding: 0.8rem 1rem;
    background: #f8fafc;
}

.product-group-header img {
    width: 40px;
    height: 40px;
    object-fit: cover;
    border-radius: 6px;
}

.product-group-header .product-title {
    font-weight: 500;
    color: #2d3748;
    font-size: 1rem;
}

.product-variants {
    padding: 0.5rem 1rem;
}

.variant-row {
    display: grid;
    grid-template-columns: 2fr 1fr 1fr 1fr;
    align-items: center;
    padding: 0.5rem 1rem;
    font-size: 0.9rem;
}

.variant-row:not(:last-child) {
    border-bottom: 1px dashed #edf2f7;
}

.variant-info {
    color: #64748b;
}

.variant-price, .variant-quantity, .variant-total {
    text-align: center;
}

.variant-total {
    font-weight: 500;
    color: #2d3748;
}

/* Payment Status Badges */
.badge {
    padding: 0.35em 0.65em;
    font-size: 0.85em;
    font-weight: 500;
    border-radius: 4px;
    display: inline-block;
    line-height: 1;
}

.badge-warning {
    background-color: #fff3cd;
    color: #856404;
    border: 1px solid #ffeeba;
}

.badge-success {
    background-color: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.badge-danger {
    background-color: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

/* Payment Details Styling */
.payment-details {
    background: #fff;
    border-radius: 8px;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05);
    margin-bottom: 2rem;
}

.payment-details h3 {
    color: #1a202c;
    font-size: 1.25rem;
    font-weight: 600;
    padding: 1.5rem;
    margin: 0;
    border-bottom: 1px solid #e2e8f0;
}

.payment-details-content {
    padding: 1.5rem;
}

.payment-info-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 0.75rem 0;
    border-bottom: 1px solid #f1f5f9;
}

.payment-info-row:last-child {
    border-bottom: none;
}

.payment-info-label {
    color: #64748b;
    font-size: 0.95rem;
}

.payment-info-value {
    color: #1a202c;
    font-weight: 500;
    font-size: 0.95rem;
}

@media (max-width: 768px) {
    .invoice-meta {
        grid-template-columns: 1fr;
        gap: 1rem;
        padding: 0 1rem;
    }
    
    .shipping-grid {
        grid-template-columns: 1fr;
    }
    
    .shipping-details {
        grid-template-columns: 1fr;
    }
    
    .order-items th:nth-child(3),
    .order-items td:nth-child(3) {
        display: none;
    }
    
    .item-image {
        width: 50px;
        height: 50px;
    }

    .invoice-top {
        grid-template-columns: 1fr;
        gap: 1rem;
        text-align: center;
    }

    .company-logo, .invoice-title, .invoice-meta {
        text-align: center;
    }

    .invoice-meta {
        align-items: center;
    }

    .meta-item {
        width: 100%;
        justify-content: space-between;
    }

    .variant-row {
        grid-template-columns: 1fr 1fr;
        gap: 0.5rem;
    }

    .variant-price {
        display: none;
    }

    .variant-quantity, .variant-total {
        text-align: right;
    }

    .invoice-header {
        padding: 1.5rem;
    }
    
    .meta-item {
        flex-direction: column;
        gap: 0.25rem;
    }
    
    .meta-label {
        min-width: auto;
    }
}
</style>
@endsection

@section("content")
<div class="mn-breadcrumb mb-4">
    <div class="row">
        <div class="col-12">
            <div class="row gi_breadcrumb_inner">
                <div class="col-md-6 col-sm-12">
                    <h2 class="mn-breadcrumb-title">Order Invoice</h2>
                </div>
                <div class="col-md-6 col-sm-12">
                    <ul class="mn-breadcrumb-list">
                        <li class="mn-breadcrumb-item"><a href="{{ route('home') }}">Home</a></li>
                        @if(url()->previous() === route('user.orders'))
                            <li class="mn-breadcrumb-item"><a href="{{ route('user.orders') }}">My Orders</a></li>
                        @else
                            <li class="mn-breadcrumb-item"><a href="{{ route('product.cart') }}">My Cart</a></li>
                            <li class="mn-breadcrumb-item"><a href="{{ route('product.checkout') }}">Checkout</a></li>
                        @endif
                        <li class="mn-breadcrumb-item active">Order Invoice</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
</div>

<section>
    @if($orderData['show_success'])
    <div class="success-message">
        <i class="ri-checkbox-circle-line"></i>
        Thank you! Your order has been placed successfully.
    </div>
    @endif

    @if($orderData['is_in_progress'])
                <div class="payment-info mb-2">
                    <i class="ri-information-line"></i>
                    @if($orderData['payment_method'] === 'cod')
                        <span><strong>Payment Status:</strong> Pending (Cash on Delivery)</span>
                    @else
                        <span>
                            <strong>Payment Status:</strong>
                            @php $ps = strtolower($orderData['payment_status'] ?? 'pending'); @endphp
                            @if($ps === 'pending' || $ps === 'unpaid' || empty($ps))
                                Awaiting Confirmation
                            @elseif($ps === 'paid')
                                Paid Successfully
                            @elseif($ps === 'failed')
                                Payment Failed
                                <a href="{{ route('product.payment.retry', ['order_id' => $orderData['order_id']]) }}" class="ms-2" style="text-decoration: underline; color: #159abe;">
                                    Try again
                                </a>
                            @elseif($ps === 'cancelled')
                                Payment Cancelled
                                <a href="{{ route('product.payment.retry', ['order_id' => $orderData['order_id']]) }}" class="ms-2" style="text-decoration: underline; color: #159abe;">
                                    Try again
                                </a>
                            @else
                                {{ ucfirst($ps) }}
                            @endif
                            @if(!empty($orderData['payment_note']))
                                <br><small class="text-danger">{{ $orderData['payment_note'] }}</small>
                            @endif
                        </span>
                    @endif
                </div>
                @endif

    <div class="invoice-container">
        <div class="invoice-header">
            <h2 class="invoice-title">Order Invoice</h2>
            <div class="invoice-meta">
                <div class="meta-group">
                    <span class="meta-label">ORDER ID</span>
                    <span class="meta-value">{{ $orderData['order_id'] }}</span>
                </div>
                <div class="meta-group">
                    <span class="meta-label">ORDER DATE</span>
                    <span class="meta-value">{{ $orderData['order_date'] }}</span>
                </div>
                <div class="meta-group">
                    <span class="meta-label">PAYMENT METHOD</span>
                    <span class="meta-value">{{ $orderData['payment_method'] === 'cod' ? 'Cash on Delivery' : 'Digital Payment' }}</span>
                </div>
            </div>
        </div>

        <div class="invoice-content">
            <div class="invoice-section">
                <h3>Shipping Information</h3>
                <div class="shipping-grid">
                    <div class="shipping-box">
                        <h4>Contact Information</h4>
                        <div class="shipping-info">
                            <p>
                                <span>Name</span>
                                <span>{{ $orderData['shipping']['contact_name'] }}</span>
                            </p>
                            <p>
                                <span>Phone</span>
                                <span>{{ $orderData['shipping']['contact_phone'] }}</span>
                            </p>
                            @if($orderData['shipping']['whatsapp'])
                            <p>
                                <span>WhatsApp</span>
                                <span>{{ $orderData['shipping']['whatsapp'] }}</span>
                            </p>
                            @endif
                            @if($orderData['shipping']['contact_email'])
                            <p>
                                <span>Email</span>
                                <span>{{ $orderData['shipping']['contact_email'] }}</span>
                            </p>
                            @endif
                        </div>
                    </div>

                    <div class="shipping-box">
                        <h4>Delivery Address</h4>
                        <div class="shipping-info">
                            <p>
                                <span>Address</span>
                                <span>{{ $orderData['shipping']['address'] }}</span>
                            </p>
                            <p>
                                <span>City</span>
                                <span>{{ $orderData['shipping']['city'] }}</span>
                            </p>
                            <p>
                                <span>State</span>
                                <span>{{ $orderData['shipping']['state'] }}</span>
                            </p>
                            <p>
                                <span>Postal Code</span>
                                <span>{{ $orderData['shipping']['postal_code'] }}</span>
                            </p>
                        </div>
                    </div>
                </div>
            </div>

            <div class="invoice-section">
                <h3>Ordered Items</h3>
                <div class="order-items">
                    @php
                        $groupedItems = collect($orderData['cart_items'])->groupBy('id');
                    @endphp

                    @foreach($groupedItems as $productId => $variants)
                        <div class="product-group">
                            <div class="product-group-header">
                                <img src="{{ $variants->first()['image'] }}" alt="{{ $variants->first()['title'] }}">
                                <span class="product-title">{{ $variants->first()['title'] }}</span>
                            </div>
                            <div class="product-variants">
                                @foreach($variants as $item)
                                    <div class="variant-row">
                                        <div class="variant-info">{{ $item['unit'] }}</div>
                                        <div class="variant-price">৳{{ number_format($item['price']) }}</div>
                                        <div class="variant-quantity">{{ $item['quantity'] }} ×</div>
                                        <div class="variant-total">৳{{ number_format($item['total']) }}</div>
                                    </div>
                                @endforeach
                            </div>
                        </div>
                    @endforeach
                </div>

                <div class="order-summary">
                    <div class="summary-row">
                        <span>Subtotal</span>
                        <span>৳{{ number_format($orderData['totals']['subtotal']) }}</span>
                    </div>
                    <div class="summary-row">
                        <span>Delivery Charge</span>
                        <span>৳{{ number_format($orderData['totals']['delivery']) }}</span>
                    </div>
                    @if($orderData['coupon'])
                    <div class="summary-row discount">
                        <span>
                            Discount
                            @if($orderData['coupon']['type'] === 'percentage')
                            ({{ $orderData['coupon']['value'] }}%)
                            @endif
                        </span>
                        <span>-৳{{ number_format($orderData['totals']['discount']) }}</span>
                    </div>
                    @endif
                    <div class="summary-row total">
                        <span>Total Amount</span>
                        <span>৳{{ number_format($orderData['totals']['total']) }}</span>
                    </div>
                </div>

                @if($orderData['payment_method'] !== 'cod')
                <div class="payment-details">
                    <h3>Payment Details</h3>
                    <div class="payment-details-content">
                        <div class="payment-info-row">
                            <span class="payment-info-label">Payment Method</span>
                            <span class="payment-info-value">{{ $orderData['payment_method'] === 'cod' ? 'Cash on Delivery' : 'Digital Payment' }}</span>
                        </div>
                        <div class="payment-info-row">
                            <span class="payment-info-label">Sender Number</span>
                            <span class="payment-info-value">{{ $orderData['payment_sender'] }}</span>
                        </div>
                        <div class="payment-info-row">
                            <span class="payment-info-label">Transaction ID</span>
                            <span class="payment-info-value">{{ $orderData['payment_trxid'] }}</span>
                        </div>
                        <div class="payment-info-row">
                            <span class="payment-info-label">Amount Paid</span>
                            <span class="payment-info-value">৳{{ number_format($orderData['totals']['total']) }}</span>
                        </div>
                        <div class="payment-info-row">
                            <span class="payment-info-label">Status</span>
                            <span class="payment-info-value">
                                @php $ps = strtolower($orderData['payment_status'] ?? 'pending'); @endphp
                                @if($ps === 'paid')
                                    <span class="badge bg-success" style="background:#198754; color:#fff;">Paid</span>
                                @elseif($ps === 'failed')
                                    <span class="badge bg-danger" style="background:#dc3545; color:#fff;">Failed</span>
                                @elseif($ps === 'cancelled')
                                    <span class="badge bg-secondary" style="background:#6c757d; color:#fff;">Cancelled</span>
                                @elseif($ps === 'verified')
                                    <span class="badge bg-success" style="background:#198754; color:#fff;">Verified</span>
                                @elseif($ps === 'rejected')
                                    <span class="badge bg-danger" style="background:#dc3545; color:#fff;">Rejected</span>
                                @else
                                    <span class="badge bg-warning text-dark" style="background:#ffc107; color:#212529;">Pending</span>
                                @endif
                            </span>
                        </div>
                    </div>
                </div>
                @endif
            </div>
        </div>
    </div>

    <div class="action-buttons">
        <a href="{{ route('home') }}" class="mn-btn-2">            
            <span><i class="ri-shopping-cart-2-line me-1"></i>Continue Shopping</span>
        </a>
        <a href="{{ route('user.orders') }}" class="mn-btn-2">
            <span><i class="ri-file-list-3-line me-1"></i>My Orders</span>
        </a>
        <a href="{{ route('product.order.track') }}" class="mn-btn-2">
            <span><i class="ri-truck-line me-1"></i>Track Your Order</span>
        </a>
        
    </div>
</section>
@endsection

@section("script")
<script>
    // Clear cart if payment was successful
    document.addEventListener('DOMContentLoaded', function() {
        @if(session('clear_cart') || $orderData['show_success'])
        // Clear the cart in localStorage
        localStorage.setItem('sb_cart', '[]');
        
        // Update cart UI
        if (typeof updateCartCount === 'function') {
            updateCartCount();
        }
        if (typeof updateCartButtons === 'function') {
            updateCartButtons();
        }
        if (typeof dispatchCartUpdate === 'function') {
            dispatchCartUpdate();
        }
        console.log('Cart cleared after successful order');
        @endif
    });
</script>
@endsection