<?php

namespace App\Services;

class EnvUpdater
{
    public static function set(string $key, string $value): void
    {
        $path = base_path('.env');
        if (!file_exists($path)) return;
        
        $content = file_get_contents($path);
        $pattern = "/^{$key}\s*=.*$/m";
        
        // Treat only truly empty strings as removal; '0' should be preserved
        if (trim($value) === '') {
            // Remove the line if value is empty
            $content = preg_replace($pattern, '', $content);
        } else {
            $escaped = self::escapeValue($value);
            if (preg_match($pattern, $content)) {
                $content = preg_replace($pattern, "{$key}={$escaped}", $content);
            } else {
                $content .= PHP_EOL . "{$key}={$escaped}";
            }
        }
        file_put_contents($path, $content);
    }

    public static function bulkSet(array $pairs): void
    {
        foreach ($pairs as $key => $value) {
            // Normalize booleans for .env: store as 1/0 for consistency
            if (is_bool($value)) {
                $value = $value ? '1' : '0';
            }
            self::set($key, (string) $value);
        }
    }

    protected static function escapeValue(string $value): string
    {
        if (preg_match('/\s/', $value) || str_contains($value, '#')) {
            $escaped = '"' . str_replace('"', '\\"', $value) . '"';
            return $escaped;
        }
        return $value;
    }
} 