<?php

namespace App\Http\Controllers;

use App\Models\Category;
use App\Models\Product;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;

class PcBuilderController extends Controller
{
    public function index()
    {
        // Components to display in the builder UI
        $components = [
            ['key' => 'cpu', 'label' => 'CPU', 'required' => true],
            ['key' => 'cpu_cooler', 'label' => 'CPU Cooler', 'required' => false],
            ['key' => 'motherboard', 'label' => 'Motherboard', 'required' => true],
            ['key' => 'ram', 'label' => 'RAM', 'required' => true],
            ['key' => 'storage', 'label' => 'Storage', 'required' => true],
            ['key' => 'gpu', 'label' => 'Graphics Card', 'required' => false],
            ['key' => 'psu', 'label' => 'Power Supply', 'required' => false],
            ['key' => 'casing', 'label' => 'Casing', 'required' => false],
            // Peripherals (optional in MVP)
            ['key' => 'monitor', 'label' => 'Monitor', 'required' => false],
            ['key' => 'keyboard', 'label' => 'Keyboard', 'required' => false],
            ['key' => 'mouse', 'label' => 'Mouse', 'required' => false],
        ];

        // Build component->categories map so UI can restrict category choices
        $componentCategories = [];
        foreach ($components as $c) {
            $componentCategories[$c['key']] = $this->getCategoriesForComponent($c['key']);
        }

        return view('general.pc_builder', [
            'components' => $components,
            'componentCategories' => $componentCategories,
        ]);
    }

    public function products(Request $request)
    {
        $request->validate([
            'component' => 'required|string',
            'category_id' => 'nullable|integer|exists:categories,id',
            'q' => 'nullable|string|max:100',
            'page' => 'nullable|integer|min:1',
            'per_page' => 'nullable|integer|min:1|max:50',
        ]);

        $component = (string) $request->get('component', '');
        $categoryId = $request->get('category_id');
        $search = (string) $request->get('q', '');
        $perPage = (int) ($request->get('per_page', 5));

        $query = Product::with([
            'brand',
            'variations' => function ($q) {
                $q->where('default', true);
            },
            'images' => function ($q) {
                $q->where('is_primary', true);
            }
        ])->select('products.*');

        // Determine relevant categories for the selected component
        $relevantCategoryIds = $this->getCategoryIdsForComponent($component);
        if (empty($relevantCategoryIds)) {
            // No relevant mapping found -> return empty
            return response()->json(['items' => [], 'meta' => ['total' => 0, 'per_page' => $perPage, 'current_page' => 1, 'last_page' => 1]]);
        }

        // Always restrict to relevant categories
        $query->whereIn('category_id', $relevantCategoryIds);

        // If a specific category within the relevant set is chosen, narrow further
        if ($categoryId && in_array((int) $categoryId, $relevantCategoryIds, true)) {
            $query->where('category_id', $categoryId);
        }

        if (!empty($search)) {
            $query->where('title', 'like', "%{$search}%");
        }

        $products = $query->latest()->paginate($perPage);

        $items = $products->map(function ($product) {
            $variation = $product->variations->first();
            return [
                'id' => (string) $product->id,
                'title' => (string) $product->title,
                'brand' => $product->brand?->title ?? '',
                'price' => $variation?->sale_price ?? 0,
                'regular_price' => $variation?->regular_price ?? 0,
                'in_stock' => (int) ($variation?->in_stock ?? 0),
                'default_variant_id' => (string) ($variation?->id ?? ''),
                'image' => $product->images->first()?->url ?? asset('assets/default_img.png'),
                'url' => route('product.view', $product->slug),
            ];
        })->values();

        return response()->json([
            'items' => $items,
            'meta' => [
                'total' => $products->total(),
                'per_page' => $products->perPage(),
                'current_page' => $products->currentPage(),
                'last_page' => $products->lastPage(),
            ],
        ]);
    }

    private function getCategoryIdsForComponent(string $component): array
    {
        $categories = $this->getCategoriesForComponent($component);
        return collect($categories)->pluck('id')->map(fn ($v) => (int) $v)->all();
    }

    private function getCategoriesForComponent(string $component): array
    {
        $component = strtolower($component);

        $likeMap = [
            'cpu' => ['processor', 'cpu'],
            'cpu_cooler' => ['cooler', 'cpu cooler'],
            'motherboard' => ['motherboard', 'mainboard'],
            'ram' => ['ram', 'memory'],
            'storage' => ['storage', 'ssd', 'hdd', 'hard disk', 'hard drive', 'nvme'],
            'gpu' => ['graphics', 'gpu', 'graphics card', 'vga', 'video card'],
            'psu' => ['power supply', 'psu'],
            'casing' => ['casing', 'case', 'chassis'],
            'monitor' => ['monitor', 'display'],
            'keyboard' => ['keyboard'],
            'mouse' => ['mouse'],
        ];

        $terms = $likeMap[$component] ?? [];
        if (empty($terms)) {
            return [];
        }

        $cats = Category::select('id', 'name')
            ->where(function ($q) use ($terms) {
                foreach ($terms as $term) {
                    $q->orWhere('name', 'like', "%{$term}%");
                }
            })
            ->orWhereHas('parent', function ($p) use ($terms) {
                $p->where(function ($q) use ($terms) {
                    foreach ($terms as $term) {
                        $q->orWhere('name', 'like', "%{$term}%");
                    }
                });
            })
            ->orderBy('name')
            ->get()
            ->map(function ($c) {
                return ['id' => $c->id, 'name' => $c->name];
            })
            ->values()
            ->all();

        return $cats;
    }
}


