<?php

namespace App\Http\Controllers;

use App\Models\Blogs;
use App\Models\BlogComment;
use App\Models\BlogReaction;
use App\Mail\Admin\NewBlogCommentNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;

class BlogController extends Controller
{
    public function index()
    {
        $blogs = Blogs::with('author')
            ->withCount(['comments', 'reactions as likes_count' => function($query) {
                $query->where('reaction_type', 'like');
            }])
            ->orderBy('created_at', 'desc')
            ->paginate(9);

        return view('blog.list', compact('blogs'));
    }

    public function show($slug)
    {
        $blog = Blogs::with(['author', 'comments' => function($query) {
                $query->whereNull('parent_id')
                    ->with(['user', 'replies.user'])
                    ->orderBy('created_at', 'desc');
            }])
            ->where('slug', $slug)
            ->firstOrFail();

        // Get user's reaction if logged in
        $userReaction = null;
        if (Auth::check()) {
            $userReaction = $blog->reactions()
                ->where('user_id', Auth::id())
                ->value('reaction_type');
        }

        // Get recent blogs except current one
        $recentBlogs = Blogs::where('id', '!=', $blog->id)
            ->orderBy('created_at', 'desc')
            ->take(5)
            ->get();

        return view('blog.post', [
            'blog' => $blog,
            'userReaction' => $userReaction,
            'recentBlogs' => $recentBlogs
        ]);
    }

    public function react(Request $request, Blogs $blog)
    {
        $request->validate([
            'type' => 'required|in:like,love,happy,angry,sad'
        ]);

        // Remove any existing reaction from this user
        $blog->reactions()->where('user_id', auth()->id())->delete();

        // Add new reaction
        $blog->reactions()->create([
            'user_id' => auth()->id(),
            'reaction_type' => $request->type
        ]);

        // Get counts for all reaction types
        $reactions = [];
        foreach (BlogReaction::REACTIONS as $type => $data) {
            $reactions[$type] = $blog->reactions()->where('reaction_type', $type)->count();
        }

        return response()->json([
            'success' => true,
            'reactions' => $reactions
        ]);
    }

    public function removeReaction(Blogs $blog)
    {
        $blog->reactions()->where('user_id', auth()->id())->delete();

        // Get counts for all reaction types
        $reactions = [];
        foreach (BlogReaction::REACTIONS as $type => $data) {
            $reactions[$type] = $blog->reactions()->where('reaction_type', $type)->count();
        }

        return response()->json([
            'success' => true,
            'reactions' => $reactions
        ]);
    }

    public function comment(Request $request, Blogs $blog)
    {
        $request->validate([
            'comment' => 'required|string|max:1000'
        ]);

        $comment = $blog->comments()->create([
            'user_id' => Auth::id(),
            'comment' => $request->comment
        ]);

        $comment->load('user');

        // Send admin notification email
        Mail::to(env('ADMIN_NOTIFIER_MAIL'))
            ->queue(new NewBlogCommentNotification($comment));

        return response()->json([
            'comment' => view('blog.partials.comment', ['comment' => $comment])->render()
        ]);
    }

    public function reply(Request $request, BlogComment $comment)
    {
        $request->validate([
            'comment' => 'required|string|max:1000'
        ]);

        $reply = BlogComment::create([
            'blog_id' => $comment->blog_id,
            'user_id' => Auth::id(),
            'parent_id' => $comment->id,
            'comment' => $request->comment
        ]);

        $reply->load('user');

        // Send admin notification email
        Mail::to(env('ADMIN_NOTIFIER_MAIL'))
            ->queue(new NewBlogCommentNotification($reply));

        return response()->json([
            'reply' => view('blog.partials.reply', ['reply' => $reply])->render()
        ]);
    }

    public function deleteComment(BlogComment $comment)
    {
        // Check if user owns the comment or is admin
        if (Auth::id() !== $comment->user_id && !Auth::user()->isAdmin()) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        // Delete the comment and its replies
        $comment->replies()->delete();
        $comment->delete();

        return response()->json(['success' => true]);
    }
} 