<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Coupons;
use Illuminate\Http\Request;
use Carbon\Carbon;

class CouponsController extends Controller
{
    public function index(Request $request)
    {
        if ($request->has('success')) {
            session()->flash('success', $request->query('success'));
        }
        if ($request->has('error')) {
            session()->flash('error', $request->query('error'));
        }

        $coupons = Coupons::latest()->get();
        return view('admin.coupons.list', compact('coupons'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'code' => 'required|string|unique:coupons,code|max:50',
            'type' => 'required|in:fixed,percentage',
            'value' => 'required|numeric|min:0',
            'valid_from' => 'required|date',
            'valid_until' => 'required|date|after:valid_from',
            'usage_limit' => 'nullable|integer|min:1',
            'usage_count' => 'nullable|integer|min:0'
        ]);

        try {
            // Format dates
            $data = $request->all();
            $data['valid_from'] = Carbon::parse($request->valid_from);
            $data['valid_until'] = Carbon::parse($request->valid_until);
            
            // Validate percentage value
            if ($request->type === 'percentage' && $request->value > 100) {
                return response()->json([
                    'success' => false,
                    'message' => 'Percentage discount cannot exceed 100%'
                ], 422);
            }

            Coupons::create($data);
            return response()->json([
                'success' => true,
                'message' => 'Coupon created successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error creating coupon'
            ], 500);
        }
    }

    public function update(Request $request, Coupons $coupon)
    {
        $request->validate([
            'code' => 'required|string|max:50|unique:coupons,code,' . $coupon->id,
            'type' => 'required|in:fixed,percentage',
            'value' => 'required|numeric|min:0',
            'valid_from' => 'required|date',
            'valid_until' => 'required|date|after:valid_from',
            'usage_limit' => 'nullable|integer|min:1',
            'usage_count' => 'nullable|integer|min:0'
        ]);

        try {
            // Format dates
            $data = $request->all();
            $data['valid_from'] = Carbon::parse($request->valid_from);
            $data['valid_until'] = Carbon::parse($request->valid_until);

            // Validate percentage value
            if ($request->type === 'percentage' && $request->value > 100) {
                return response()->json([
                    'success' => false,
                    'message' => 'Percentage discount cannot exceed 100%'
                ], 422);
            }

            $coupon->update($data);
            return response()->json([
                'success' => true,
                'message' => 'Coupon updated successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating coupon'
            ], 500);
        }
    }

    public function destroy(Coupons $coupon)
    {
        try {
            // Don't allow deleting active coupons that have been used
            if ($coupon->usage_count > 0 && Carbon::now()->between($coupon->valid_from, $coupon->valid_until)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete an active coupon that has been used'
                ], 422);
            }

            $coupon->delete();
            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting coupon'
            ], 500);
        }
    }
} 