@extends('admin_base')

@section('title', 'Edit Product')

@section('style')
<link href="https://cdn.jsdelivr.net/npm/summernote@0.8.18/dist/summernote-bs4.min.css" rel="stylesheet">
<style>
.form-label {
    font-weight: 600;
}
.required:after {
    content: ' *';
    color: red;
}
.image-preview {
    width: 150px;
    height: 150px;
    border: 2px dashed #ddd;
    border-radius: 8px;
    margin: 10px;
    position: relative;
    overflow: hidden;
    display: inline-flex;
    align-items: center;
    justify-content: center;
}
.image-preview img {
    width: 100%;
    height: 100%;
    object-fit: contain;
    padding: 5px;
}
.image-preview-container {
    position: relative;
    display: inline-block;
    margin: 10px;
}
.image-preview-footer {
    margin-top: 5px;
    width: 150px;
}
.image-preview-footer input {
    width: 100%;
    padding: 4px;
    border: 1px solid #ced4da;
    border-radius: 4px;
    font-size: 0.8rem;
}
.image-preview .remove-image {
    position: absolute;
    top: 5px;
    right: 5px;
    background: rgba(220, 53, 69, 0.8);
    color: white;
    border: none;
    border-radius: 50%;
    width: 25px;
    height: 25px;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
}
.image-preview .remove-image:hover {
    background: rgba(220, 53, 69, 1);
}
.image-preview .primary-badge {
    position: absolute;
    bottom: 5px;
    left: 5px;
    background: rgba(25, 135, 84, 0.9);
    color: white;
    padding: 2px 8px;
    border-radius: 12px;
    font-size: 0.75rem;
}
.image-preview .make-primary {
    position: absolute;
    bottom: 5px;
    left: 5px;
    background: rgba(0, 0, 0, 0.5);
    color: white;
    padding: 2px 8px;
    border-radius: 12px;
    font-size: 0.75rem;
    cursor: pointer;
    opacity: 0;
    transition: opacity 0.2s;
}
.image-preview:hover .make-primary {
    opacity: 1;
}
.image-upload {
    width: 150px;
    height: 150px;
    border: 2px dashed #159abe;
    border-radius: 8px;
    margin: 10px;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    color: #159abe;
    transition: all 0.3s ease;
}
.image-upload:hover {
    background-color: rgba(58, 78, 229, 0.1);
}
.variant-card {
    border: 1px solid #e0e0e0;
    border-radius: 8px;
    padding: 20px;
    margin-bottom: 15px;
    position: relative;
    background-color: #f8f9fa;
    box-shadow: 0 2px 4px rgba(0,0,0,0.05);
    transition: all 0.2s ease;
}

.variant-card.out-of-stock {
    background-color: #fde7e9;
    border-color: #f5c2c7;
}

.variant-card.critical-stock {
    background-color: #fff3f4;
    border-color: #fecdd3;
}

.variant-card.low-stock {
    background-color: #fff8e1;
    border-color: #ffe082;
}

.variant-card .remove-variant {
    position: absolute;
    top: 10px;
    right: 10px;
    color: #dc3545;
    cursor: pointer;
}

.variant-card.out-of-stock .remove-variant,
.variant-card.critical-stock .remove-variant {
    color: rgba(255,255,255,0.9);
}
.tag-input {
    display: flex;
    flex-wrap: wrap;
    gap: 8px;
    padding: 8px;
    border: 1px solid #ced4da;
    border-radius: 4px;
    min-height: 45px;
    background: #fff;
}
.tag {
    background-color: #e9ecef;
    border: 1px solid #dee2e6;
    border-radius: 20px;
    padding: 4px 12px;
    display: inline-flex;
    align-items: center;
    gap: 8px;
    font-size: 0.9rem;
    transition: all 0.2s;
}
.tag:hover {
    background-color: #dee2e6;
}
.tag .remove-tag {
    color: #dc3545;
    cursor: pointer;
    font-size: 1.1rem;
    line-height: 1;
    padding: 2px;
}
.tag .remove-tag:hover {
    color: #bb2d3b;
}
.tag-input input {
    border: none;
    outline: none;
    padding: 4px;
    flex-grow: 1;
    min-width: 120px;
    font-size: 0.9rem;
}
.tag-suggestions {
    position: absolute;
    top: 100%;
    left: 0;
    right: 0;
    background: white;
    border: 1px solid #ced4da;
    border-radius: 4px;
    margin-top: 4px;
    max-height: 200px;
    overflow-y: auto;
    z-index: 1000;
    display: none;
}
.tag-suggestion {
    padding: 8px 12px;
    cursor: pointer;
    transition: background-color 0.2s;
}
.tag-suggestion:hover {
    background-color: #f8f9fa;
}
.section-card {
    background-color: #ffffff;
    border: 1px solid #e0e0e0;
    border-radius: 8px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.05);
    margin-bottom: 20px;
}
.section-card .card-header {
    background-color: #f8f9fa;
    border-bottom: 1px solid #e0e0e0;
    padding: 15px 20px;
}
.section-card .card-body {
    padding: 20px;
}
.variant-default-label {
    display: flex;
    align-items: center;
    gap: 8px;
    cursor: pointer;
    padding: 8px;
    border-radius: 4px;
    transition: background-color 0.2s;
}
.variant-default-label:hover {
    background-color: #e9ecef;
}
div.note-modal .form-group.note-form-group.note-group-select-from-files {
    display: none;
}
</style>
@endsection

@section('content')
        <form id="productForm" onsubmit="updateProduct(event)">
            <div class="row">
                <div class="col-md-8">
                    <!-- Basic Information -->
            <div class="section-card">
                <div class="card-header">
                    <h5 class="mb-0">Basic Information</h5>
                </div>
                        <div class="card-body">
                    <div class="form-group mb-3">
                        <label for="title" class="form-label required">Product Title</label>
                                <input type="text" class="form-control" id="title" name="title" 
                                       value="{{ $product->title }}" required>
                            </div>

                    <div class="form-group mb-3">
                        <label for="category_id" class="form-label required">Category</label>
                                <select class="form-control" id="category_id" name="category_id" required>
                                    <option value="">Select Category</option>
                                    @foreach($categories as $category)
                                        <option value="{{ $category->id }}" 
                                            {{ $category->id === $product->category_id ? 'selected' : '' }}>
                                    {{ $category->full_path }}
                                        </option>
                                    @endforeach
                                </select>
                            </div>
                    <div class="form-group mb-3">
                        <label for="brand_id" class="form-label">Brand (optional)</label>
                        <select class="form-control" id="brand_id" name="brand_id">
                            <option value="">None</option>
                            @foreach(\App\Models\Brand::orderBy('title')->get() as $b)
                                <option value="{{ $b->id }}" {{ $product->brand && $product->brand->id === $b->id ? 'selected' : '' }}>{{ $b->title }}</option>
                            @endforeach
                        </select>
                    </div>

                    <div class="form-group mb-3">
                        <label for="description" class="form-label required">Description</label>
                        <textarea class="form-control summernote" id="description" name="description" rows="4" required>{{ $product->description }}</textarea>
                    </div>

                    <!-- Specification Section -->
                    <div class="form-group mb-3">
                        <label class="form-label">Specification</label>
                        <div id="specificationList"></div>
                        <button type="button" class="btn btn-outline-primary btn-sm mt-2" onclick="addSpecification()">
                            <i class="fas fa-plus"></i> Add Specification
                        </button>
                    </div>

                    <!-- Features Section -->
                    <div class="form-group mb-3">
                        <label class="form-label">Features</label>
                        <div id="featuresList"></div>
                        <button type="button" class="btn btn-outline-primary btn-sm mt-2" onclick="addFeature()">
                            <i class="fas fa-plus"></i> Add Feature
                        </button>
                    </div>

                    <div class="form-group mb-3">
                        <label class="form-label">Tags</label>
                        <div class="position-relative">
                                <div class="tag-input" onclick="document.getElementById('tagInput').focus()">
                                    <div id="tagContainer"></div>
                                <input type="text" id="tagInput" placeholder="Type and press Enter to add tags...">
                            </div>
                            <div id="tagSuggestions" class="tag-suggestions"></div>
                        </div>
                        <small class="text-muted">Press Enter or comma to add a tag</small>
                    </div>
                        </div>
                    </div>

                    <!-- Variants -->
            <div class="section-card mt-4">
                <div class="card-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">Product Variants</h5>
                        <button type="button" class="sb_btn btn-sm" onclick="addVariant()">
                                    <i class="fas fa-plus"></i> Add Variant
                                </button>
                            </div>
                </div>
                <div class="card-body">
                            <div id="variantsContainer">
                                <!-- Variants will be added here -->
                            </div>
                        </div>
                    </div>
                </div>

                <div class="col-md-4">
                    <!-- Images -->
            <div class="section-card">
                <div class="card-header">
                    <h5 class="mb-0">Product Images</h5>
                </div>
                        <div class="card-body">
                    <div id="imagePreviewContainer" class="mb-3">
                                <!-- Image previews will be added here -->
                            </div>
                            <label class="image-upload" for="imageInput">
                        <i class="fas fa-plus"></i> Add Images
                            </label>
                            <input type="file" id="imageInput" accept="image/*" style="display: none" multiple>
                    <small class="text-muted d-block mt-2">Click an image to set as primary.</small>
                        </div>
                    </div>

                    <!-- SEO Information -->
            <div class="section-card mt-4">
                <div class="card-header">
                    <h5 class="mb-0">SEO Information</h5>
                </div>
                        <div class="card-body">
                    <div class="form-group mb-3">
                        <label for="meta_title" class="form-label">Meta Title</label>
                                <input type="text" class="form-control" id="meta_title" name="meta_title" 
                                       value="{{ $product->meta_title }}">
                                <small class="text-muted">Leave empty to use product title</small>
                            </div>

                    <div class="form-group mb-3">
                        <label for="meta_description" class="form-label">Meta Description</label>
                                <textarea class="form-control" id="meta_description" name="meta_description" 
                                          rows="3">{{ $product->meta_description }}</textarea>
                                <small class="text-muted">Leave empty to use product description</small>
                            </div>

                    <div class="form-group mb-3">
                        <label for="meta_keywords" class="form-label">Meta Keywords</label>
                                <input type="text" class="form-control" id="meta_keywords" name="meta_keywords" 
                                       value="{{ $product->meta_keywords }}">
                                <small class="text-muted">Leave empty to use tags</small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

    <div class="row mt-4 mb-5 justify-content-center">
        <div class="col-12 text-center mb-3">
            <button type="submit" class="sb_btn">
                <i class="fas fa-save"></i> Update Product
            </button>
            <a href="{{ route('admin.products') }}" class="btn btn-secondary">
                <i class="fas fa-times"></i> Cancel
            </a>
        </div>
    </div>
        </form>
@endsection

@section('script')
<script src="https://cdn.jsdelivr.net/npm/summernote@0.8.18/dist/summernote-bs4.min.js"></script>
<script>
// Initialize Summernote
$(document).ready(function() {
    $('.summernote').summernote({
        height: 300,
        toolbar: [
            ['style', ['style']],
            ['font', ['bold', 'underline', 'clear']],
            ['color', ['color']],
            ['para', ['ul', 'ol', 'paragraph']],
            ['table', ['table']],
            ['insert', ['link', 'picture', 'video']],
            ['view', ['fullscreen', 'codeview', 'help']]
        ]
    });
});

const productData = {
    tags: {!! json_encode($product->tags ?? []) !!},
    images: {!! json_encode($product->images->map(function($img) {
        return [
            'id' => $img->id,
            'path' => $img->path,
            'alt_text' => $img->alt_text ?? '',
            'is_primary' => (bool)$img->is_primary,
            'sort_order' => $img->sort_order ?? 0
        ];
    })->all()) !!},
    variations: {!! json_encode($product->variations ?? []) !!},
    specification: {!! json_encode($product->specification ?? []) !!},
    features: {!! json_encode($product->features ?? []) !!}
};

let tags = productData.tags;
let images = productData.images;
let variants = productData.variations.map(v => ({
    ...v,
    id: v.id,
    default: Boolean(v.default)
}));

// Make sure at least one variant is default
if (!variants.some(v => v.default) && variants.length > 0) {
    variants[0].default = true;
}
let primaryImageIndex = images.findIndex(img => img.is_primary) || 0;
let specification = productData.specification;

// Add features management
let features = productData.features || [];

function addFeature(text = '') {
    if (features.length >= 5) {
        return;
    }
    features.push(text);
    renderFeatures();
}

function removeFeature(index) {
    features.splice(index, 1);
    renderFeatures();
}

function updateFeature(index, value) {
    features[index] = value;
}

function renderFeatures() {
    const container = document.getElementById('featuresList');
    if (!container) return;
    
    container.innerHTML = features.map((feature, idx) => `
        <div class="row mb-2 align-items-center">
            <div class="col-10">
                <input type="text" 
                       class="form-control" 
                       placeholder="Enter product feature" 
                       value="${feature}"
                       onchange="updateFeature(${idx}, this.value)">
            </div>
            <div class="col-2 text-end">
                <button type="button" class="btn btn-danger btn-sm" onclick="removeFeature(${idx})">&times;</button>
            </div>
        </div>
    `).join('');

    // Show/hide the add button based on features count
    const addButton = document.querySelector('button[onclick="addFeature()"]');
    if (addButton) {
        if (features.length >= 5) {
            addButton.style.display = 'none';
            // Add message about max features
            let maxMessage = container.querySelector('.max-features-message');
            if (!maxMessage) {
                maxMessage = document.createElement('small');
                maxMessage.className = 'text-muted d-block mt-2 max-features-message';
                maxMessage.textContent = 'Maximum 5 features reached';
                container.parentNode.insertBefore(maxMessage, addButton);
            }
        } else {
            addButton.style.display = 'inline-block';
            // Remove max message if exists
            const maxMessage = container.parentNode.querySelector('.max-features-message');
            if (maxMessage) {
                maxMessage.remove();
            }
        }
    }
}

// Tags Management
const tagInput = document.getElementById('tagInput');
const tagSuggestions = document.getElementById('tagSuggestions');
let commonTags = ['featured', 'new', 'sale', 'popular', 'trending'];

tagInput.addEventListener('keydown', function(e) {
    if ((e.key === 'Enter' || e.key === ',') && !e.shiftKey) {
        e.preventDefault();
        const tag = this.value.trim().toLowerCase();
        addTag(tag);
    }
});

tagInput.addEventListener('input', function() {
    const value = this.value.trim().toLowerCase();
    if (value) {
        const suggestions = commonTags.filter(tag => 
            tag.toLowerCase().includes(value) && !tags.includes(tag)
        );
        if (suggestions.length > 0) {
            tagSuggestions.innerHTML = suggestions.map(tag => `
                <div class="tag-suggestion" onclick="addTag('${tag}')">${tag}</div>
            `).join('');
            tagSuggestions.style.display = 'block';
        } else {
            tagSuggestions.style.display = 'none';
        }
    } else {
        tagSuggestions.style.display = 'none';
    }
});

document.addEventListener('click', function(e) {
    if (!tagInput.contains(e.target) && !tagSuggestions.contains(e.target)) {
        tagSuggestions.style.display = 'none';
        }
});

function addTag(tag) {
    tag = tag.trim().toLowerCase();
    if (tag && !tags.includes(tag)) {
        tags.push(tag);
        renderTags();
        tagInput.value = '';
        tagSuggestions.style.display = 'none';
    }
}

function removeTag(index) {
    tags.splice(index, 1);
    renderTags();
}

function renderTags() {
    const container = document.getElementById('tagContainer');
    container.innerHTML = tags.map((tag, index) => `
        <div class="tag">
            ${tag}
            <span class="remove-tag" onclick="removeTag(${index})">&times;</span>
        </div>
    `).join('');
}

// Image Management
document.getElementById('imageInput').addEventListener('change', function(e) {
    const files = Array.from(e.target.files).slice(0, 10); // Limit to 10 images
    if (files.length > 10) {
        alert('You can upload maximum 10 images');
        return;
    }
    
    files.forEach(file => {
        if (file.size > 5 * 1024 * 1024) { // 5MB limit
            alert(`Image ${file.name} is too large. Maximum size is 5MB`);
            return;
        }
        
        const reader = new FileReader();
        reader.onload = function(e) {
            images.push({
                id: null,
                path: e.target.result,
                alt_text: '', // Remove default alt text
                is_primary: false
            });
            renderImages();
        }
        reader.readAsDataURL(file);
    });
});

function removeImage(index) {
    images.splice(index, 1);
    if (primaryImageIndex >= images.length) {
        primaryImageIndex = Math.max(0, images.length - 1);
    }
    renderImages();
}

function updateImageAlt(index, value) {
    if (images[index]) {
        images[index].alt_text = value;
    }
}

function setPrimaryImage(index) {
    primaryImageIndex = index;
    renderImages();
}

function renderImages() {
    const container = document.getElementById('imagePreviewContainer');
    container.innerHTML = images.map((image, index) => `
        <div class="image-preview-container">
            <div class="image-preview">
                <img src="${image.path.startsWith('data:') ? image.path : '{{ asset("storage/") }}/' + image.path}" 
                     alt="${image.alt_text || 'Preview'}">
                <button type="button" class="remove-image" onclick="removeImage(${index})">&times;</button>
                ${index === primaryImageIndex 
                    ? '<span class="primary-badge">Primary</span>'
                    : `<span class="make-primary" onclick="setPrimaryImage(${index})">Set as Primary</span>`
                }
            </div>
            <div class="image-preview-footer">
                <input type="text" 
                       placeholder="Alt text" 
                       value="${image.alt_text || ''}"
                       onchange="updateImageAlt(${index}, this.value)"
                       class="form-control form-control-sm">
            </div>
        </div>
    `).join('');
}

// Variants Management
function updateVariant(id, field, value) {
    const variant = variants.find(v => String(v.id) === String(id));
    if (variant) {
        if (field === 'default') {
            value = Boolean(value);
            // Unset default from all other variants
            variants.forEach(v => v.default = false);
            variant[field] = value;
            renderVariants(); // Only re-render for default changes
        } else if (['sale_price', 'regular_price', 'in_stock'].includes(field)) {
            // Convert numeric fields to numbers
            variant[field] = Number(value);
        } else {
            variant[field] = value;
        }
    }
}

function setDefaultVariant(id) {
    variants.forEach(v => {
        v.default = String(v.id) === String(id);
    });
    renderVariants();
}

function addVariant() {
    const variant = {
        id: 'new_' + Date.now(),
        unit_type: '',
        unit_value: '',
        sale_price: '',
        regular_price: '',
        in_stock: '',
        default: variants.length === 0 // First variant is default only if no variants exist
    };
    variants.push(variant);
    renderVariants();
}

function removeVariant(id) {
    const index = variants.findIndex(v => String(v.id) === String(id));
    if (index === -1) return;
    
    const wasDefault = variants[index].default;
    variants.splice(index, 1);
    
    // If we removed the default variant and there are other variants,
    // make the first remaining variant default
    if (wasDefault && variants.length > 0) {
        variants[0].default = true;
    }
    
    renderVariants();
}

function renderVariants() {
    const container = document.getElementById('variantsContainer');
    container.innerHTML = variants.map(variant => {
        let stockClass = '';
        // Only add stock status classes if in_stock is defined (not for new variants)
        if (variant.in_stock !== undefined && variant.in_stock !== '') {
            if (variant.in_stock <= 0) {
                stockClass = 'out-of-stock';
            } else if (variant.in_stock <= 5) {
                stockClass = 'critical-stock';
            } else if (variant.in_stock <= 10) {
                stockClass = 'low-stock';
            }
        }
        return `
        <div class="variant-card ${stockClass}">
            <i class="fas fa-times remove-variant" onclick="removeVariant('${variant.id}')"></i>
            <div class="row">
                <div class="col-md-6">
                    <div class="form-group mb-3">
                        <label class="form-label required">Unit Type</label>
                        <input type="text" class="form-control" value="${variant.unit_type || ''}"
                               onchange="updateVariant('${variant.id}', 'unit_type', this.value)" required>
                    </div>
                </div>
                <div class="col-md-6">
                    <div class="form-group mb-3">
                        <label class="form-label required">Unit Value</label>
                        <input type="text" class="form-control" value="${variant.unit_value || ''}"
                               onchange="updateVariant('${variant.id}', 'unit_value', this.value)" required>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="form-group mb-3">
                        <label class="form-label required">Sale Price</label>
                        <input type="number" class="form-control" value="${variant.sale_price !== undefined ? variant.sale_price : ''}"
                               onchange="updateVariant('${variant.id}', 'sale_price', this.value)" required>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="form-group mb-3">
                        <label class="form-label required">Regular Price</label>
                        <input type="number" class="form-control" value="${variant.regular_price !== undefined ? variant.regular_price : ''}"
                               onchange="updateVariant('${variant.id}', 'regular_price', this.value)" required>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="form-group mb-3">
                        <label class="form-label required">Stock</label>
                        <input type="number" class="form-control" value="${variant.in_stock !== undefined ? variant.in_stock : ''}"
                               onchange="updateVariant('${variant.id}', 'in_stock', this.value)" required>
                    </div>
                </div>
                <div class="col-12">
                    <label class="variant-default-label">
                        <input type="radio" name="default_variant" 
                               ${variant.default ? 'checked' : ''} 
                               onclick="setDefaultVariant('${variant.id}')"
                               ${variants.length === 1 ? 'disabled' : ''}>
                        Set as Default Variant ${variants.length === 1 ? '(Only Variant)' : ''}
                    </label>
                </div>
            </div>
        </div>
    `}).join('');
}

// Specification Management
function addSpecification(key = '', value = '') {
    specification.push({key, value});
    renderSpecification();
}

function removeSpecification(index) {
    specification.splice(index, 1);
    renderSpecification();
}

function updateSpecification(index, field, val) {
    specification[index][field] = val;
}

function renderSpecification() {
    const container = document.getElementById('specificationList');
    if (!container) return;
    container.innerHTML = specification.map((item, idx) => `
        <div class="row mb-2 align-items-center">
            <div class="col-5">
                <input type="text" class="form-control" placeholder="Key" value="${item.key}" onchange="updateSpecification(${idx}, 'key', this.value)">
            </div>
            <div class="col-5">
                <input type="text" class="form-control" placeholder="Value" value="${item.value}" onchange="updateSpecification(${idx}, 'value', this.value)">
            </div>
            <div class="col-2 text-end">
                <button type="button" class="btn btn-danger btn-sm" onclick="removeSpecification(${idx})">&times;</button>
            </div>
        </div>
    `).join('');
}

// Form Submission
function updateProduct(event) {
    event.preventDefault();
    
    if (variants.length === 0) {
        alert('Please add at least one variant');
        return;
    }
    
    if (images.length === 0) {
        alert('Please add at least one image');
        return;
    }

    // Reorder images array to put primary image first
    const reorderedImages = [...images];
    if (primaryImageIndex > 0) {
        const primaryImage = reorderedImages.splice(primaryImageIndex, 1)[0];
        reorderedImages.unshift(primaryImage);
    }

    // Ensure variants data is properly formatted
    const formattedVariants = variants.map(v => ({
        ...v,
        id: v.id.toString().startsWith('new_') ? null : v.id,
        default: Boolean(v.default) // Ensure default is boolean
    }));

    // Ensure at least one variant is default
    if (!formattedVariants.some(v => v.default)) {
        formattedVariants[0].default = true;
    }

    const formData = {
        title: document.getElementById('title').value,
        category_id: document.getElementById('category_id').value,
        brand_id: document.getElementById('brand_id').value || null,
        description: document.getElementById('description').value,
        tags: tags,
        images: reorderedImages.map(img => ({
            path: img.path,
            alt_text: img.alt_text || ''
        })),
        meta_title: document.getElementById('meta_title').value || null,
        meta_description: document.getElementById('meta_description').value || null,
        meta_keywords: document.getElementById('meta_keywords').value || null,
        variants: formattedVariants,
        specification: specification.filter(item => item.key && item.value),
        features: features.filter(feature => feature.trim() !== '') // Add features to form data
    };

    // Show loading state
    const submitBtn = event.target.querySelector('button[type="submit"]');
    const originalText = submitBtn.innerHTML;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Updating...';
    submitBtn.disabled = true;

    fetch('/admin/products/{{ $product->id }}', {
        method: 'PUT',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}',
            'Accept': 'application/json'
        },
        body: JSON.stringify(formData)
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Store success message in session storage
            sessionStorage.setItem('flash_message', 'Product updated successfully');
            sessionStorage.setItem('flash_type', 'success');
            
            // Redirect to products page
            window.location.href = data.redirect;
        } else {
            alert(data.message || 'Error updating product');
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error updating product');
        submitBtn.innerHTML = originalText;
        submitBtn.disabled = false;
    });
}

// Initialize
renderTags();
renderImages();
renderVariants();
renderSpecification();
renderFeatures();
</script>
@endsection