@extends('admin_base')

@section('title', 'Order Details')

@section('style')
<style>
.order-container {
    background: #fff;
    border-radius: 12px;
    box-shadow: 0 2px 20px rgba(0, 0, 0, 0.06);
    margin-bottom: 2rem;
}

.order-header {
    background: linear-gradient(to right, #159abe, #4e6ae5);
    padding: 2rem;
    border-radius: 12px 12px 0 0;
    color: white;
}

.order-title {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 2rem;
}

.order-title h2 {
    color: white;
    font-size: 1.5rem;
    font-weight: 600;
    margin: 0;
}

.order-meta {
    display: grid;
    grid-template-columns: repeat(4, 1fr);
    gap: 2rem;
    background: white;
    padding: 2rem;
    border-radius: 8px;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
    margin: 2rem;
}

.meta-group {
    text-align: center;
    padding: 1.5rem;
    background: #f8fafc;
    border-radius: 8px;
    border: 1px solid #e2e8f0;
}

.meta-label {
    color: #6b7280;
    font-size: 0.875rem;
    text-transform: uppercase;
    letter-spacing: 0.05em;
    margin-bottom: 0.5rem;
}

.meta-value {
    color: #1a202c;
    font-size: 1.1rem;
    font-weight: 600;
}

.order-content {
    padding: 2rem;
}

.order-section {
    margin-bottom: 2.5rem;
    background: white;
    border-radius: 8px;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
    overflow: hidden;
}

.order-section h3 {
    color: #2d3748;
    font-size: 1.2rem;
    font-weight: 600;
    margin: 0;
    padding: 1.2rem;
    background: #f8fafc;
    border-bottom: 2px solid #edf2f7;
}

.info-grid {
    display: grid;
    grid-template-columns: repeat(2, 1fr);
    gap: 2rem;
    padding: 1.5rem;
}

.info-table {
    width: 100%;
    background: white;
    border-radius: 8px;
    border: 1px solid #e2e8f0;
    overflow: hidden;
}

.info-table th {
    background: #f8fafc;
    padding: 1rem 1.5rem;
    text-align: left;
    font-weight: 600;
    color: #2d3748;
    border-bottom: 1px solid #e2e8f0;
}

.info-table td {
    padding: 0.75rem 1.5rem;
    border-bottom: 1px solid #f1f5f9;
}

.info-table tr:last-child td {
    border-bottom: none;
}

.info-table td:first-child {
    color: #64748b;
    width: 120px;
    font-size: 0.95rem;
}

.info-table td:last-child {
    color: #1a202c;
    font-weight: 500;
}

.info-box {
    background: #f8fafc;
    padding: 1.5rem;
    border-radius: 8px;
    border: 1px solid #edf2f7;
}

.info-box h4 {
    color: #2d3748;
    font-size: 1rem;
    font-weight: 600;
    margin-bottom: 1.2rem;
    padding-bottom: 0.8rem;
    border-bottom: 1px solid #e2e8f0;
}

.info-list {
    display: grid;
    gap: 1rem;
}

.info-item {
    display: grid;
    grid-template-columns: 120px 1fr;
    align-items: center;
    padding: 0.75rem;
    background: white;
    border-radius: 6px;
    transition: all 0.2s;
}

.info-item:hover {
    background: #f1f5f9;
}

.info-label {
    color: #64748b;
    font-size: 0.9rem;
}

.info-value {
    color: #1a202c;
    font-weight: 500;
}

.order-items {
    padding: 1.5rem;
}

.product-group {
    background: white;
    border-radius: 8px;
    margin-bottom: 1rem;
    border: 1px solid #edf2f7;
    overflow: hidden;
}

.product-header {
    display: flex;
    align-items: center;
    gap: 1rem;
    padding: 1rem;
    background: #f8fafc;
    border-bottom: 1px solid #edf2f7;
}

.product-image {
    width: 60px;
    height: 60px;
    object-fit: cover;
    border-radius: 8px;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

.product-title {
    font-weight: 600;
    color: #2d3748;
    font-size: 1.1rem;
}

.variants-list {
    padding: 0;
}

.variant-item {
    display: grid;
    grid-template-columns: 1fr auto auto;
    align-items: center;
    gap: 1.5rem;
    padding: 0.75rem 1rem;
    border-bottom: 1px solid #f1f5f9;
}

.variant-item:last-child {
    border-bottom: none;
}

.variant-info {
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
}

.variant-name {
    color: #64748b;
    font-size: 0.9rem;
}

.variant-price {
    color: #4a5568;
    font-weight: 500;
}

.variant-quantity {
    color: #4a5568;
    font-weight: 500;
    padding: 0.5rem 1rem;
    background: #f1f5f9;
    border-radius: 6px;
    text-align: center;
}

.variant-total {
    font-weight: 600;
    color: #2d3748;
    font-size: 1.1rem;
}

.order-summary {
    padding: 1.5rem;
}

.summary-row {
    display: flex;
    justify-content: space-between;
    padding: 0.8rem;
    color: #4a5568;
    font-size: 1rem;
    background: #f8fafc;
    border-radius: 6px;
    margin-bottom: 0.5rem;
}

.summary-row.discount {
    background: #f0f9ff;
    color: #0369a1;
    border: 1px solid #bae6fd;
    padding: 1rem 1.5rem;
    margin-bottom: 0;
    border-radius: 8px 8px 0 0;
}

.coupon-details {
    background: #f0f9ff;
    padding: 1rem 1.5rem;
    border-radius: 0 0 8px 8px;
    margin: 0 0 0.5rem 0;
    border: 1px solid #bae6fd;
    border-top: none;
}

.coupon-info {
    display: grid;
    grid-template-columns: repeat(3, 1fr);
    gap: 1rem;
}

.coupon-info p {
    margin: 0;
    text-align: center;
    padding: 0.5rem;
    background: rgba(255, 255, 255, 0.5);
    border-radius: 6px;
}

.coupon-info p span {
    display: block;
}

.coupon-info p span:first-child {
    color: #0369a1;
    font-size: 0.8rem;
    text-transform: uppercase;
    margin-bottom: 0.25rem;
}

.coupon-info p span:last-child {
    color: #0c4a6e;
    font-weight: 500;
}

/* Add new styles for payment section */
.payment-status {
    display: inline-flex;
    align-items: center;
    padding: 0.5rem 1rem;
    border-radius: 6px;
    font-weight: 500;
    gap: 0.5rem;
}

.payment-status.pending {
    background: #fff3cd;
    color: #856404;
    border: 1px solid #ffeeba;
}

.payment-status.verified {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.payment-status.rejected {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.payment-info {
    display: grid;
    gap: 1rem;
    padding: 1.5rem;
}

.payment-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1rem;
}

.payment-actions {
    display: flex;
    gap: 0.5rem;
}

.payment-details {
    display: grid;
    grid-template-columns: repeat(2, 1fr);
    gap: 1.5rem;
    background: #f8fafc;
    padding: 1.5rem;
    border-radius: 8px;
}

.payment-detail-group {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.payment-label {
    color: #64748b;
    font-size: 0.875rem;
    font-weight: 500;
}

.payment-value {
    color: #1a202c;
    font-size: 1rem;
    font-weight: 600;
}

/* Improved timeline styles */
.timeline {
    padding: 2rem;
}

.timeline-item {
    position: relative;
    padding-left: 3rem;
    padding-bottom: 2rem;
}

.timeline-item:last-child {
    padding-bottom: 0;
}

.timeline-marker {
    position: absolute;
    left: 0;
    top: 0;
    width: 1.5rem;
    height: 1.5rem;
    border-radius: 50%;
    background-color: #fff;
    border: 2px solid #159abe;
    box-shadow: 0 0 0 4px rgba(58, 78, 229, 0.1);
    z-index: 1;
}

.timeline-item:not(:last-child):before {
    content: '';
    position: absolute;
    left: 0.65rem;
    top: 1.5rem;
    bottom: 0;
    width: 2px;
    background-color: #e2e8f0;
}

.timeline-content {
    background: #f8fafc;
    padding: 1.5rem;
    border-radius: 8px;
    border: 1px solid #e2e8f0;
}

.timeline-title {
    color: #1a202c;
    font-weight: 600;
    font-size: 1.1rem;
    margin: 0;
    margin-bottom: 0.5rem;
}

.timeline-date {
    color: #64748b;
    font-size: 0.9rem;
    margin: 0;
}

.action-buttons {
    display: flex;
    gap: 1rem;
}

.action-buttons .btn {
    padding: 0.5rem 1rem;
    font-weight: 500;
}

@media (max-width: 768px) {
    .order-meta {
        grid-template-columns: repeat(2, 1fr);
        gap: 1rem;
    }
    
    .info-grid {
        grid-template-columns: 1fr;
    }
    
    .variant-row {
        grid-template-columns: 1fr 1fr;
        gap: 0.5rem;
    }
    
    .variant-price {
        display: none;
    }
    
    .variant-quantity, .variant-total {
        text-align: right;
    }
}
</style>
@endsection

@section('content')
<div class="row">
    <div class="col-12">
        <div class="order-container">
            <div class="order-header">
                <div class="order-title">
                    <h2>Order #{{ $order->order_number }}</h2>
                    <div class="action-buttons">
                        <a href="{{ route('admin.orders') }}" class="btn btn-light">
                            <i class="fas fa-arrow-left me-1"></i> Back to Orders
                        </a>
                        <a href="{{ route('admin.orders.invoice', $order) }}" 
                           class="btn btn-info"
                           onclick="event.preventDefault(); printInvoice(this.href);">
                            <i class="fas fa-file-invoice me-1"></i> Print Invoice
                        </a>
                    </div>
                </div>
            </div>

            <div class="order-meta">
                <div class="meta-group">
                    <span class="meta-label">Order Date</span>
                    <div class="mt-2">
                        <span class="meta-value">{{ $order->created_at->format('d M, Y h:i A') }}</span>
                    </div>
                </div>
                <div class="meta-group">
                    <span class="meta-label">Placed By</span>
                    <div class="mt-2">
                        <a href="{{ route('admin.users.edit', $order->user_id) }}" class="text-primary meta-value">
                            {{ $order->user->name }}
                        </a>
                    </div>
                </div>
                <div class="meta-group">
                    <span class="meta-label">Payment Info</span>
                    <div class="mt-2">
                        <span class="meta-value">{{ strtoupper($order->payment_method) }} - {{ ucfirst($order->payment_status ?? 'unpaid') }}</span>
                    </div>
                </div>
                <div class="meta-group">
                    <span class="meta-label">Order Status</span>
                    <div class="dropdown mt-2">
                        <button class="btn btn-{{ $order->order_status === 'pending' ? 'warning' : ($order->order_status === 'processing' ? 'info' : ($order->order_status === 'shipped' ? 'primary' : ($order->order_status === 'delivered' ? 'success' : 'danger'))) }} dropdown-toggle" 
                                type="button" data-toggle="dropdown">
                            {{ ucfirst($order->order_status) }}
                        </button>
                        <div class="dropdown-menu">
                            @foreach($statuses as $status => $label)
                                @if($status !== $order->order_status)
                                <form action="{{ route('admin.orders.status', $order) }}" method="POST" class="d-inline">
                                    @csrf
                                    @method('PUT')
                                    <input type="hidden" name="status" value="{{ $status }}">
                                    <button type="submit" class="dropdown-item">
                                        Mark as {{ $label }}
                                    </button>
                                </form>
                                @endif
                            @endforeach
                        </div>
                    </div>
                </div>
            </div>

            <div class="order-content">
                <div class="order-section">
                    <h3>Customer Information</h3>
                    <div class="info-grid">
                        <table class="info-table">
                            <thead>
                                <tr>
                                    <th colspan="2">Contact Details</th>
                                </tr>
                            </thead>
                            <tbody>
                                <tr>
                                    <td>Name</td>
                                    <td>{{ $order->contact_name }}</td>
                                </tr>
                                <tr>
                                    <td>Phone</td>
                                    <td>{{ $order->contact_phone }}</td>
                                </tr>
                                @if($order->whatsapp)
                                <tr>
                                    <td>WhatsApp</td>
                                    <td>{{ $order->whatsapp }}</td>
                                </tr>
                                @endif
                                @if($order->contact_email)
                                <tr>
                                    <td>Email</td>
                                    <td>{{ $order->contact_email }}</td>
                                </tr>
                                @endif
                            </tbody>
                        </table>

                        <table class="info-table">
                            <thead>
                                <tr>
                                    <th colspan="2">Shipping Address</th>
                                </tr>
                            </thead>
                            <tbody>
                                <tr>
                                    <td>Address</td>
                                    <td>{{ $order->address }}</td>
                                </tr>
                                <tr>
                                    <td>City</td>
                                    <td>{{ $order->city }}</td>
                                </tr>
                                <tr>
                                    <td>State</td>
                                    <td>{{ $order->state }}</td>
                                </tr>
                                <tr>
                                    <td>Postal Code</td>
                                    <td>{{ $order->postal_code }}</td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>

                <div class="order-section">
                    <h3>Order Items</h3>
                    <div class="order-items">
                        @php
                            $groupedItems = $order->items->groupBy('product_id');
                        @endphp
                        
                        @foreach($groupedItems as $productId => $items)
                            @php
                                $firstItem = $items->first();
                            @endphp
                            <div class="product-group">
                                <div class="product-header">
                                    <img src="{{ $firstItem->product->primary_image->url }}" alt="{{ $firstItem->product->title }}" class="product-image">
                                    <span class="product-title">{{ $firstItem->product->title }}</span>
                                </div>
                                <div class="variants-list">
                                    @foreach($items as $item)
                                    <div class="variant-item">
                                        <div class="variant-info">
                                            <span class="variant-name">{{ $item->variation->unit_value }}{{ $item->variation->unit_type }}</span>
                                            <span class="variant-price">৳{{ number_format($item->getUnitPrice()) }} each</span>
                                        </div>
                                        <span class="variant-quantity">{{ $item->quantity }} ×</span>
                                        <span class="variant-total">৳{{ number_format($item->getTotal()) }}</span>
                                    </div>
                                    @endforeach
                                </div>
                            </div>
                        @endforeach
                    </div>

                    <div class="order-summary">
                        <div class="summary-row">
                            <span>Subtotal</span>
                            <span>৳{{ number_format($order->subtotal) }}</span>
                        </div>
                        <div class="summary-row">
                            <span>Delivery Charge</span>
                            <span>৳{{ number_format($order->delivery_charge) }}</span>
                        </div>
                        @if($order->discount > 0)
                            <div class="summary-row discount">
                                <span>Discount</span>
                                <span>-৳{{ number_format($order->discount) }}</span>
                            </div>
                            @if($order->coupon_code)
                            <div class="coupon-details">
                                <div class="coupon-info">
                                    <p>
                                        <span>Coupon Code</span>
                                        <span>{{ $order->coupon_code }}</span>
                                    </p>
                                    <p>
                                        <span>Type</span>
                                        <span>{{ ucfirst($order->coupon_type) }}</span>
                                    </p>
                                    <p>
                                        <span>Value</span>
                                        <span>
                                            @if($order->coupon_type === 'percentage')
                                                {{ $order->coupon_value }}%
                                            @else
                                                ৳{{ number_format($order->coupon_value) }}
                                            @endif
                                        </span>
                                    </p>
                                </div>
                            </div>
                            @endif
                        @endif
                        <div class="summary-row total">
                            <span>Total Amount</span>
                            <span>৳{{ number_format($order->total) }}</span>
                        </div>
                    </div>
                </div>

                <div class="order-section">
                    <h3>Payment Details</h3>
                    <div class="payment-info">
                        <div class="payment-details">
                            <div class="payment-detail-group">
                                <span class="payment-label">Payment Method</span>
                                <span class="payment-value">{{ strtoupper($order->payment_method) }}</span>
                            </div>
                            <div class="payment-detail-group">
                                <span class="payment-label">Status</span>
                                <span class="payment-value">{{ ucfirst($order->payment_status ?? 'unpaid') }}</span>
                            </div>
                            @if($order->payment_trxid)
                            <div class="payment-detail-group">
                                <span class="payment-label">Transaction</span>
                                <span class="payment-value">{{ $order->payment_trxid }}</span>
                            </div>
                            @endif
                        </div>
                    </div>
                </div>

                <div class="order-section">
                    <h3>Order Timeline</h3>
                    <div class="timeline">
                        @php
                            $allEvents = collect();
                            
                            // Add order statuses
                            $order->statuses->each(function($status) use ($allEvents) {
                                // Format the status text to be more user-friendly
                                $statusText = match($status->status) {
                                    'pending' => 'Order Placed',
                                    'processing' => 'Processing Order',
                                    'shipped' => 'Order Shipped',
                                    'delivered' => 'Order Delivered',
                                    'cancelled' => 'Order Cancelled',
                                    'payment_pending' => 'Payment Pending',
                                    'payment_verified' => 'Payment Verified',
                                    'payment_rejected' => 'Payment Rejected',
                                    default => ucfirst($status->status)
                                };

                                $allEvents->push([
                                    'type' => 'order',
                                    'status' => $status->status,
                                    'display_text' => $statusText,
                                    'date' => $status->created_at,
                                    'note' => null
                                ]);
                            });
                            
                            // Add payment status if exists
                            // Use new order payment fields if available
                            if ($order->payment_status) {
                                $paymentText = match($order->payment_status) {
                                    'unpaid' => 'Payment Pending',
                                    'paid' => 'Payment Verified',
                                    'failed' => 'Payment Failed',
                                    'cancelled' => 'Payment Cancelled',
                                    default => ucfirst($order->payment_status) . ' Payment'
                                };

                                $allEvents->push([
                                    'type' => 'payment',
                                    'status' => $order->payment_status,
                                    'display_text' => $paymentText,
                                    'date' => $order->created_at,
                                    'note' => null
                                ]);
                            }
                            
                            // Sort all events by date, newest first
                            $allEvents = $allEvents->sortByDesc('date');
                        @endphp

                        @foreach($allEvents as $event)
                        <div class="timeline-item">
                            <div class="timeline-marker"></div>
                            <div class="timeline-content">
                                <h6 class="timeline-title">
                                    @if($event['type'] === 'order')
                                        <i class="fas fa-{{ 
                                            $event['status'] === 'pending' ? 'clock' : 
                                            ($event['status'] === 'processing' ? 'cog' : 
                                            ($event['status'] === 'shipped' ? 'truck' : 
                                            ($event['status'] === 'delivered' ? 'check-circle' : 'times-circle'))) 
                                        }} me-2"></i>
                                    @else
                                        @php
                                            $icon = 'wallet';
                                            if ($event['status'] === 'paid' || $event['status'] === 'verified') {
                                                $icon = 'check-circle';
                                            } elseif ($event['status'] === 'failed' || $event['status'] === 'rejected' || $event['status'] === 'cancelled') {
                                                $icon = 'times-circle';
                                            }
                                        @endphp
                                        <i class="fas fa-{{ $icon }} me-2"></i>
                                    @endif
                                    {{ $event['display_text'] }}
                                </h6>
                                <p class="timeline-date">{{ $event['date']->format('d M, Y h:i A') }}</p>
                                @if($event['note'])
                                <div class="mt-2 text-muted small">
                                    <i class="fas fa-info-circle me-1"></i> {{ $event['note'] }}
                                </div>
                                @endif
                            </div>
                        </div>
                        @endforeach
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Payment Approval Modal -->
@if(false)
<div class="modal fade" id="approvePaymentModal" tabindex="-1">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header bg-light">
                <h5 class="modal-title">
                    <i class="fas fa-check-circle text-success me-2"></i>
                    Confirm Payment Approval
                </h5>
                <button type="button" class="close" data-dismiss="modal">
                    <span>&times;</span>
                </button>
            </div>
            <form action="{{ route('orders.payment.approve', $order) }}" method="POST">
                @csrf
                <div class="modal-body">
                    <div class="alert alert-warning mb-3">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        Please verify payment details before approving.
                    </div>

                    <table class="table table-sm">
                        <tr>
                            <td class="text-muted" width="35%">Payment Method</td>
                            <td>{{ $order->payment->paymentMethod->name }}</td>
                        </tr>
                        <tr>
                            <td class="text-muted">Order Total</td>
                            <td>৳{{ number_format($order->total) }}</td>
                        </tr>
                        <tr>
                            <td class="text-muted">Amount Paid</td>
                            <td>
                                ৳{{ number_format($order->payment->amount) }}
                                <span class="ms-2 small">
                                    @php
                                        $diff = $order->payment->amount - $order->total;
                                    @endphp
                                    @if($diff < 0)
                                        <span class="text-danger">(৳{{ number_format(abs($diff)) }} less)</span>
                                    @elseif($diff > 0)
                                        <span class="text-warning">(৳{{ number_format($diff) }} more)</span>
                                    @else
                                        <span class="text-success">(Exact amount)</span>
                                    @endif
                                </span>
                            </td>
                        </tr>
                        <tr>
                            <td class="text-muted">Transaction ID</td>
                            <td><span class="font-monospace">{{ $order->payment->transaction_id }}</span></td>
                        </tr>
                    </table>
                </div>
                <div class="modal-footer bg-light">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-success">✓ Approve</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Payment Rejection Modal -->
<div class="modal fade" id="rejectPaymentModal" tabindex="-1">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header bg-light">
                <h5 class="modal-title">
                    <i class="fas fa-times-circle text-danger me-2"></i>
                    Reject Payment
                </h5>
                    <button type="button" class="close" data-dismiss="modal">
                        <span>&times;</span>
                    </button>
                </div>
            <form action="{{ route('orders.payment.reject', $order) }}" method="POST">
                @csrf
                <div class="modal-body">
                    <div class="alert alert-warning mb-3">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        Please provide a reason for rejecting this payment.
                    </div>

                    <div class="preset-messages mb-3">
                        <label class="d-block mb-2">Quick Responses:</label>
                        <div class="d-flex flex-wrap gap-2">
                            <button type="button" class="btn btn-sm btn-outline-secondary preset-msg" 
                                    data-message="The amount you've sent (৳{{ number_format($order->payment->amount) }}) doesn't match the order total (৳{{ number_format($order->total) }}). Please send the correct amount.">
                                Wrong Amount
                            </button>
                            <button type="button" class="btn btn-sm btn-outline-secondary preset-msg" 
                                    data-message="We couldn't verify this transaction ID in our payment gateway. Please ensure you've sent the payment and provided the correct transaction ID.">
                                Invalid TrxID
                            </button>
                            <button type="button" class="btn btn-sm btn-outline-secondary preset-msg" 
                                    data-message="The payment was sent to the wrong merchant account. Please ensure you're sending to the correct merchant number provided during checkout.">
                                Wrong Receiver
                            </button>
                        </div>
                    </div>

                    <div class="form-group">
                        <label for="note">Rejection Note:</label>
                        <textarea class="form-control" 
                                id="note" 
                                name="note" 
                                rows="3" 
                                required 
                                placeholder="Type or select a preset message above"></textarea>
                    </div>
                </div>
                <div class="modal-footer bg-light">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-danger">✕ Reject</button>
                </div>
            </form>
        </div>
    </div>
</div>
@endif

@endsection 

@section('script')
<script>
function printInvoice(url) {
    const iframe = document.createElement('iframe');
    iframe.style.display = 'none';
    document.body.appendChild(iframe);
    iframe.src = url;
    iframe.onload = function() {
        iframe.contentWindow.print();
        setTimeout(function() {
            document.body.removeChild(iframe);
        }, 1000);
    };
}

function approvePayment() {
    $('#approvePaymentModal').modal('show');
}

function rejectPayment() {
    $('#rejectPaymentModal').modal('show');
}

// Preset message handler
$(document).on('click', '.preset-msg', function() {
    const message = $(this).data('message');
    $('#note').val(message);
});
</script>
@endsection  