@extends('admin_base')

@section('title', 'Coupons')

@section('style')
<style>
.coupon-badge {
    padding: 5px 10px;
    border-radius: 15px;
    font-size: 0.85rem;
}
.coupon-badge.active {
    background-color: #28a745;
    color: white;
}
.coupon-badge.expired {
    background-color: #dc3545;
    color: white;
}
.coupon-badge.upcoming {
    background-color: #ffc107;
    color: #000;
}
.coupon-code {
    font-family: monospace;
    font-size: 1.1rem;
    font-weight: 600;
    color: #159abe;
}
.add-btn {
    margin-bottom: 20px;
}
.usage-count {
    font-size: 0.9rem;
    color: #6c757d;
}
.table-actions {
    display: flex;
    gap: 8px;
    justify-content: center;
}
.table-actions button {
    border: none;
    padding: 6px 12px;
    border-radius: 4px;
    cursor: pointer;
    transition: all 0.2s;
}
.edit-btn {
    background-color: #ffc107;
    color: #000;
}
.edit-btn:hover {
    background-color: #e0a800;
}
.delete-btn {
    background-color: #dc3545;
    color: white;
}
.delete-btn:hover {
    background-color: #c82333;
}
</style>
@endsection

@section('content')
<div class="card">
    <div class="card-header">
        <h3 class="card-title">Coupons</h3>
    </div>
    <div class="card-body">
        <button class="sb_btn add-btn" onclick="showAddModal()">
            <i class="fas fa-plus"></i> Add New Coupon
        </button>

        <div class="table-responsive">
            <table id="couponsTable" class="table table-bordered table-striped text-center">
                <thead class="bg_sb_primary">
                    <tr>
                        <th>#</th>
                        <th>Code</th>
                        <th>Type</th>
                        <th>Value</th>
                        <th>Valid Period</th>
                        <th>Usage</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach($coupons as $key => $coupon)
                    @php
                        $now = \Carbon\Carbon::now();
                        $status = $now->between($coupon->valid_from, $coupon->valid_until) ? 'active' : 
                                ($now->lt($coupon->valid_from) ? 'upcoming' : 'expired');
                    @endphp
                    <tr>
                        <td>{{ $key + 1 }}</td>
                        <td>
                            <span class="coupon-code">{{ $coupon->code }}</span>
                        </td>
                        <td>{{ ucfirst($coupon->type) }}</td>
                        <td>
                            @if($coupon->type === 'percentage')
                                {{ $coupon->value }}%
                            @else
                                ৳{{ number_format($coupon->value, 2) }}
                            @endif
                        </td>
                        <td>
                            <span class="text-info">{{ $coupon->valid_from->format('M j, Y') }}</span> - <span class="text-danger">{{ $coupon->valid_until->format('M j, Y') }}</span>
                        </td>
                        <td>
                            {{ $coupon->usage_count }} / 
                            {{ $coupon->usage_limit ?: '∞' }}
                            @if($coupon->usage_limit && $coupon->usage_count >= $coupon->usage_limit)
                                <br><span class="text-danger">Limit reached</span>
                            @endif
                        </td>
                        <td>
                            <span class="coupon-badge {{ $status }}">
                                {{ ucfirst($status) }}
                            </span>
                        </td>
                        <td>
                            <div class="table-actions">
                                <button class="edit-btn" onclick="showEditModal({{ $coupon->id }})">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button class="delete-btn" onclick="showDeleteModal({{ $coupon->id }}, '{{ $coupon->code }}')">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Add/Edit Coupon Modal -->
<div class="modal fade" id="couponModal" tabindex="-1" role="dialog" aria-labelledby="couponModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered" role="document">
        <div class="modal-content">
            <div class="modal-header bg_sb_primary">
                <h5 class="modal-title" id="couponModalLabel">Add New Coupon</h5>
                <button type="button" class="close text-white" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <form id="couponForm" onsubmit="saveCoupon(event)">
                <div class="modal-body">
                    <input type="hidden" id="couponId">
                    <div class="form-group">
                        <label for="code" class="required">Coupon Code</label>
                        <input type="text" class="form-control" id="code" name="code" required
                               pattern="[A-Za-z0-9]+" title="Only letters and numbers allowed">
                        <small class="form-text text-muted">Only letters and numbers, no spaces or special characters</small>
                    </div>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="type" class="required">Discount Type</label>
                                <select class="form-control" id="type" name="type" required onchange="updateValueLabel()">
                                    <option value="fixed">Fixed Amount</option>
                                    <option value="percentage">Percentage</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="value" class="required">Discount Value</label>
                                <input type="number" class="form-control" id="value" name="value" 
                                       min="0" step="0.01" required>
                                <small class="form-text text-muted" id="valueHelp">Enter amount in BDT</small>
                            </div>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="valid_from" class="required">Valid From</label>
                                <input type="date" class="form-control" id="valid_from" name="valid_from" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="valid_until" class="required">Valid Until</label>
                                <input type="date" class="form-control" id="valid_until" name="valid_until" required>
                            </div>
                        </div>
                    </div>
                    <div class="form-group">
                        <label for="usage_limit">Usage Limit</label>
                        <input type="number" class="form-control" id="usage_limit" name="usage_limit" 
                               min="1" placeholder="Leave empty for unlimited">
                    </div>
                </div>
                <div class="modal-footer d-flex justify-content-between p-1">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">
                        <i class="fas fa-times"></i> Cancel
                    </button>
                    <button type="submit" class="sb_btn">
                        <i class="fas fa-save"></i> <span id="saveButtonText">Save Coupon</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div class="modal fade" id="deleteModal" tabindex="-1" role="dialog" aria-labelledby="deleteModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered" role="document">
        <div class="modal-content">
            <div class="modal-header bg-danger text-white">
                <h5 class="modal-title" id="deleteModalLabel">Confirm Delete</h5>
                <button type="button" class="close text-white" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                Are you sure you want to delete coupon: <strong id="deleteCouponCode"></strong>?
                <br>
                <small class="text-danger">This action cannot be undone!</small>
            </div>
            <div class="modal-footer d-flex justify-content-between p-1">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">
                    <i class="fas fa-times"></i> Cancel
                </button>
                <button type="button" class="btn btn-danger" onclick="deleteCoupon()">
                    <i class="fas fa-trash"></i> Delete Coupon
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
$(document).ready(function() {
    $('#couponsTable').DataTable({
        responsive: true,
        dom: 'Bfrtip',
        buttons: [
            'copy', 'csv', 'excel', 'pdf', 'print'
        ],
        order: [[0, 'asc']],
        pageLength: 10,
        language: {
            search: "Search:",
            lengthMenu: "Show _MENU_ entries per page",
            zeroRecords: "No matching records found",
            info: "Showing _START_ to _END_ of _TOTAL_ entries",
            infoEmpty: "Showing 0 to 0 of 0 entries",
            infoFiltered: "(filtered from _MAX_ total entries)"
        }
    });

    // Set min date for valid_from to today
    const today = new Date().toISOString().split('T')[0];
    $('#valid_from').attr('min', today);

    // Update valid_until min date when valid_from changes
    $('#valid_from').change(function() {
        $('#valid_until').attr('min', $(this).val());
    });
});

let couponIdToDelete = null;

function updateValueLabel() {
    const type = $('#type').val();
    const helpText = type === 'percentage' ? 'Enter percentage (0-100)' : 'Enter amount in BDT';
    $('#valueHelp').text(helpText);
    
    // Update max value for percentage
    if (type === 'percentage') {
        $('#value').attr('max', '100');
    } else {
        $('#value').removeAttr('max');
    }
}

function showAddModal() {
    $('#couponId').val('');
    $('#couponForm')[0].reset();
    updateValueLabel();
    $('#couponModalLabel').text('Add New Coupon');
    $('#saveButtonText').text('Save Coupon');
    $('#couponModal').modal('show');
}

function showEditModal(couponId) {
    const coupons = @json($coupons);
    const couponData = coupons.find(c => c.id === couponId);
    
    $('#couponId').val(couponId);
    $('#code').val(couponData.code);
    $('#type').val(couponData.type);
    $('#value').val(couponData.value);
    $('#valid_from').val(couponData.valid_from.split('T')[0]);
    $('#valid_until').val(couponData.valid_until.split('T')[0]);
    $('#usage_limit').val(couponData.usage_limit);
    
    updateValueLabel();
    $('#couponModalLabel').text('Edit Coupon');
    $('#saveButtonText').text('Update Coupon');
    $('#couponModal').modal('show');
}

function showDeleteModal(couponId, couponCode) {
    couponIdToDelete = couponId;
    $('#deleteCouponCode').text(couponCode);
    $('#deleteModal').modal('show');
}

function saveCoupon(event) {
    event.preventDefault();
    
    const couponId = $('#couponId').val();
    const isEdit = couponId !== '';
    const url = isEdit 
        ? `/admin/coupons/${couponId}`
        : '/admin/coupons';
    const method = isEdit ? 'PUT' : 'POST';
    
    const formData = {
        code: $('#code').val().toUpperCase(),
        type: $('#type').val(),
        value: $('#value').val(),
        valid_from: $('#valid_from').val(),
        valid_until: $('#valid_until').val(),
        usage_limit: $('#usage_limit').val() || null
    };

    fetch(url, {
        method: method,
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}',
            'Accept': 'application/json'
        },
        body: JSON.stringify(formData)
    })
    .then(response => response.json())
    .then(data => {
        $('#couponModal').modal('hide');
        if (data.success) {
            window.location.href = "{{ route('admin.coupons') }}?success=" + data.message;
        } else {
            window.location.href = "{{ route('admin.coupons') }}?error=" + data.message;
        }
    })
    .catch(error => {
        $('#couponModal').modal('hide');
        window.location.href = "{{ route('admin.coupons') }}?error=Error saving coupon";
    });
}

function deleteCoupon() {
    if (couponIdToDelete) {
        fetch(`/admin/coupons/${couponIdToDelete}`, {
            method: 'DELETE',
            headers: {
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'Accept': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            $('#deleteModal').modal('hide');
            if (data.success) {
                window.location.href = "{{ route('admin.coupons') }}?success=Coupon deleted successfully";
            } else {
                window.location.href = "{{ route('admin.coupons') }}?error=" + data.message;
            }
        })
        .catch(error => {
            $('#deleteModal').modal('hide');
            window.location.href = "{{ route('admin.coupons') }}?error=Error deleting coupon";
        });
    }
}
</script>
@endsection 