@extends('admin_base')

@section('title', 'Blog Posts')

@section('style')
<style>
.blog-thumbnail {
    width: 60px;
    height: 60px;
    object-fit: cover;
    border-radius: 8px;
    background: #f8f9fa;
}
.blog-tags {
    display: flex;
    flex-wrap: wrap;
    gap: 6px;
    row-gap: 6px;
}
.blog-tag {
    background-color: #e9ecef;
    border-radius: 16px;
    padding: 2px 12px;
    font-size: 0.92em;
    color: #495057;
    margin-bottom: 0;
}
.blog-tag.more {
    background: #f5f6fa;
    color: #888;
    font-weight: 500;
    font-size: 0.92em;
    padding: 2px 10px;
}
.stats-row {
    display: flex;
    align-items: center;
    gap: 15px;
}
.stat {
    display: flex;
    align-items: center;
    gap: 5px;
    color: #666;
    font-size: 0.9rem;
}
.stat .reaction-emoji {
    font-size: 1.1rem;
    line-height: 1;
}
.stat i {
    font-size: 1rem;
    color: #666;
}
.table td, .table th {
    vertical-align: middle;
}
.btn-group .btn + .btn {
    margin-left: 8px;
}
.card { margin-bottom: 0; padding-bottom: 0; }
.table { margin-bottom: 0; }
.search-bar {
    position: relative;
    margin-bottom: 18px;
}
.search-bar input {
    width: 100%;
    padding: 10px 40px 10px 16px;
    border-radius: 8px;
    border: 1px solid #e0e0e0;
    font-size: 1rem;
}
.search-bar .fa-search {
    position: absolute;
    right: 16px;
    top: 50%;
    transform: translateY(-50%);
    color: #aaa;
}
</style>
@endsection

@section('content')
<div class="container-fluid">
    <div class="mb-3">
        <h1 class="h3 mb-0 text-gray-800">Blog Posts</h1>
    </div>
    <div class="search-bar">
        <input type="text" id="blogSearchInput" placeholder="Search posts by title or tag...">
        <i class="fas fa-search"></i>
    </div>
    <div class="card">
        <div class="card-body p-0">
            <div class="table-responsive">
                <table class="table align-middle mb-0">
                    <thead class="bg_sb_primary">
                        <tr>
                            <th style="width: 80px;">Image</th>
                            <th>Title</th>
                            <th style="width: 30%">Tags</th>
                            <th style="width: 120px; text-align: center;">Stats</th>
                            <th style="width: 160px;">Date</th>
                            <th style="width: 110px;">Actions</th>
                        </tr>
                    </thead>
                    <tbody id="blogListTableBody">
                        @forelse($blogs as $blog)
                        <tr>
                            <td>
                                <img src="{{ asset('storage/' . $blog->thumbnail) }}" alt="{{ $blog->title }}" class="blog-thumbnail">
                            </td>
                            <td>
                                <div style="font-weight:600;">{{ $blog->title }}</div>
                                    <div class="text-muted" style="font-size:0.95em;">By {{ $blog->author->first_name . ' ' . $blog->author->last_name }}</div>
                            </td>
                            <td>
                                <div class="blog-tags">
                                    @php $maxTags = 4; $tagCount = is_array($blog->tags) ? count($blog->tags) : 0; @endphp
                                    @if($blog->tags && $tagCount)
                                        @foreach(array_slice($blog->tags, 0, $maxTags) as $tag)
                                            <span class="blog-tag">{{ $tag }}</span>
                                        @endforeach
                                        @if($tagCount > $maxTags)
                                            <span class="blog-tag more">+{{ $tagCount - $maxTags }}</span>
                                        @endif
                                    @endif
                                </div>
                            </td>
                            <td>
                                <div class="stats-row">
                                    <span class="stat" title="Comments">
                                        <i class="fas fa-comments"></i> {{ $blog->comments_count }}
                                    </span>
                                    @foreach(App\Models\BlogReaction::REACTIONS as $type => $reaction)
                                    <span class="stat" title="{{ ucfirst($type) }}">
                                        <span class="reaction-emoji">{{ $reaction['emoji'] }}</span>
                                        {{ $blog->reactions()->where('reaction_type', $type)->count() }}
                                    </span>
                                    @endforeach
                                </div>
                            </td>
                            <td>
                                <div>{{ $blog->created_at->format('d M, Y h:i A') }}</div>
                                <small class="text-muted">{{ $blog->created_at->format('h:i A') }}</small>
                            </td>
                            <td>
                                <div class="btn-group">
                                    <a href="{{ route('admin.blogs.edit', $blog) }}" class="btn btn-sm btn-primary" title="Edit">
                                        <i class="fas fa-edit"></i>
                                    </a>
                                    <button type="button" class="btn btn-sm btn-danger" style="margin-left:8px;" onclick="deleteBlog({{ $blog->id }})" title="Delete">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </td>
                        </tr>
                        @empty
                        <tr>
                            <td colspan="6" class="text-center py-5 text-muted">No blog posts found.</td>
                        </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    <div class="mt-4 mb-0 px-3 pb-3">
        {{ $blogs->links('partial.pagination') }}
    </div>
</div>
@endsection

@section('script')
<script>
const searchInput = document.getElementById('blogSearchInput');
searchInput.addEventListener('input', function() {
    const filter = this.value.toLowerCase();
    const rows = document.querySelectorAll('#blogListTableBody tr');
    rows.forEach(row => {
        const title = row.querySelector('td:nth-child(2) div').textContent.toLowerCase();
        const tags = Array.from(row.querySelectorAll('.blog-tag')).map(t => t.textContent.toLowerCase()).join(' ');
        if (title.includes(filter) || tags.includes(filter)) {
            row.style.display = '';
        } else {
            row.style.display = 'none';
        }
    });
});
function deleteBlog(id) {
    if (!confirm('Are you sure you want to delete this blog post? This will also delete all comments and reactions.')) {
        return;
    }
    fetch(`/admin/blogs/${id}`, {
        method: 'DELETE',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            window.location.reload();
        } else {
            alert(data.message || 'Error deleting blog post');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error deleting blog post');
    });
}
</script>
@endsection