/**
 * Product Compare Functionality
 */

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    initCompareButtons();
    updateCompareCount();
});

/**
 * Initialize compare buttons on product cards
 */
function initCompareButtons() {
    const compareButtons = document.querySelectorAll('.mn-compare');
    
    compareButtons.forEach(button => {
        const productId = button.dataset.productId;
        
        // Check if product is already in compare list
        if (isInCompare(productId)) {
            button.classList.add('compared');
        } else {
            button.classList.remove('compared');
        }
        
        // Add click event listener
        button.addEventListener('click', function(e) {
            e.preventDefault();
            toggleCompare(productId, button);
        });
    });
}

/**
 * Toggle product in compare list
 */
function toggleCompare(productId, button) {
    if (isInCompare(productId)) {
        removeFromCompare(productId);
        syncCompareButtons(productId, false);
        showNotification('Product removed from compare list', 'info');
    } else {
        addToCompare(productId);
        syncCompareButtons(productId, true);
        showNotification('Product added to compare list', 'success');
    }
    
    // Update compare count
    updateCompareCount();
}

/**
 * Update all compare buttons for a given product id
 */
function syncCompareButtons(productId, isCompared) {
    const buttons = document.querySelectorAll(`.mn-compare[data-product-id="${productId}"]`);
    buttons.forEach(btn => {
        if (isCompared === undefined) {
            // derive from storage
            if (isInCompare(productId)) {
                btn.classList.add('compared');
            } else {
                btn.classList.remove('compared');
            }
        } else if (isCompared) {
            btn.classList.add('compared');
        } else {
            btn.classList.remove('compared');
        }
    });
}

/**
 * Check if product is in compare list
 */
function isInCompare(productId) {
    const compareProducts = getCompareProducts();
    return compareProducts.includes(productId.toString());
}

/**
 * Add product to compare list
 */
function addToCompare(productId) {
    const compareProducts = getCompareProducts();
    
    // Add new product
    compareProducts.push(productId.toString());
    
    // Save to localStorage
    localStorage.setItem('compareProducts', JSON.stringify(compareProducts));
}

/**
 * Remove product from compare list
 */
function removeFromCompare(productId) {
    let compareProducts = getCompareProducts();
    compareProducts = compareProducts.filter(id => id !== productId.toString());
    localStorage.setItem('compareProducts', JSON.stringify(compareProducts));
}

/**
 * Get compare products from localStorage
 */
function getCompareProducts() {
    const compareProducts = localStorage.getItem('compareProducts');
    return compareProducts ? JSON.parse(compareProducts) : [];
}

/**
 * Update compare count in header
 */
function updateCompareCount() {
    const compareProducts = getCompareProducts();
    const uniqueCount = Array.from(new Set(compareProducts.map(id => id.toString()))).length;
    const compareCountElements = document.querySelectorAll('.compare-count, .mn-compare-count');
    
    compareCountElements.forEach(element => {
        element.textContent = uniqueCount;
    });
}