<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;

class Product extends Model
{
    protected $fillable = [
        'category_id',
        'brand_id',
        'title',
        'slug',
        'description',
        'specification',
        'features',
        'tags',
        'meta_title',
        'meta_description',
        'meta_keywords'
    ];

    protected $casts = [
        'specification' => 'array',
        'tags' => 'array',
        'features' => 'array'
    ];

    protected static function boot()
    {
        parent::boot();
        static::creating(function ($product) {
            $product->slug = Str::slug($product->title);
            
            if (empty($product->meta_title)) {
                $product->meta_title = $product->title;
            }
            
            if (empty($product->meta_description)) {
                $product->meta_description = Str::limit(strip_tags($product->description), 160);
            }
            
            if (empty($product->meta_keywords)) {
                $keywords = collect($product->tags ?? []);
                if ($product->category) {
                    $keywords->push($product->category->name);
                }
                $product->meta_keywords = $keywords->join(', ');
            }
        });
    }

    public function category(): BelongsTo
    {
        return $this->belongsTo(Category::class);
    }

    public function variations(): HasMany
    {
        return $this->hasMany(ProductVariation::class);
    }

    public function brand(): BelongsTo
    {
        return $this->belongsTo(Brand::class);
    }

    public function images(): HasMany
    {
        return $this->hasMany(ProductImage::class)->orderBy('sort_order');
    }

    public function ratings(): HasMany
    {
        return $this->hasMany(ProductRating::class);
    }

    public function getAverageRatingAttribute(): float
    {
        $avg = $this->ratings()->avg('rating');
        return $avg ? round((float)$avg, 2) : 0.0;
    }

    public function getRatingsCountAttribute(): int
    {
        return (int) $this->ratings()->count();
    }

    /**
     * Get the primary image of the product
     */
    public function getPrimaryImageAttribute()
    {
        return $this->images()->where('is_primary', true)->first() 
            ?? $this->images()->first();
    }

    /**
     * Get minimum price from variations
     */
    public function getMinPriceAttribute(): int
    {
        return $this->variations()->min('price') ?? 0;
    }

    /**
     * Get SEO meta title (fallback to product title)
     */
    public function getMetaTitleAttribute($value): string
    {
        return $value ?: $this->title;
    }

    /**
     * Get SEO meta description (fallback to truncated product description)
     */
    public function getMetaDescriptionAttribute($value): ?string
    {
        return $value ?: Str::limit(strip_tags($this->description), 160);
    }

    /**
     * Get full image URL for og:image
     */
    public function getOgImageUrlAttribute(): ?string
    {
        return $this->primary_image ? $this->primary_image->url : null;
    }

    
} 