<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class BestDeals extends Model
{
    protected $fillable = [
        'name',
        'description',
        'start_date',
        'end_date',
        'status'
    ];

    protected $casts = [
        'start_date' => 'datetime',
        'end_date' => 'datetime',
        'status' => 'boolean'
    ];

    /**
     * Get the products associated with the deal.
     */
    public function products(): BelongsToMany
    {
        return $this->belongsToMany(Product::class, 'best_deal_product')
                    ->withTimestamps();
    }

    /**
     * Scope a query to only include active deals.
     */
    public function scopeActive($query)
    {
        return $query->where('status', true)
                    ->where('start_date', '<=', now())
                    ->where('end_date', '>=', now());
    }

    /**
     * Deactivate all other deals except the current one
     */
    public static function deactivateOtherDeals($exceptDealId)
    {
        static::where('id', '!=', $exceptDealId)
              ->where('status', true)
              ->update(['status' => false]);
    }

    /**
     * Override the save method to handle single active deal logic
     */
    public function save(array $options = [])
    {
        $isNew = !$this->exists;
        $statusChanged = $this->isDirty('status') && $this->status;

        $result = parent::save($options);

        // If this deal is being set to active, deactivate all others
        if (($isNew || $statusChanged) && $this->status) {
            static::deactivateOtherDeals($this->id);
        }

        return $result;
    }
} 