<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Category;
use App\Models\ProductImage;
use App\Models\ProductVariation;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use App\Models\ProductRating;

class ProductsController extends Controller
{
    public function index(Request $request)
    {
        $query = Product::with(['category', 'variations', 'images'])
            ->when($request->search, function($q) use ($request) {
                $search = $request->search;
                return $q->where('title', 'like', "%{$search}%")
                    ->orWhereHas('category', function($q) use ($search) {
                        $q->where('name', 'like', "%{$search}%");
                    })
                    ->orWhere(function($q) use ($search) {
                        $q->whereRaw('JSON_CONTAINS(tags, \'["' . $search . '"]\')');
                    });
            });

        $products = $query->latest()->paginate(20)->withQueryString();
        
        if ($request->ajax()) {
            return response()->json([
                'html' => view('admin.products._product_cards', compact('products'))->render(),
                'hasMorePages' => $products->hasMorePages()
            ]);
        }

        return view('admin.products.list', compact('products'));
    }

    public function create()
    {
        $categories = Category::all();
        return view('admin.products.create', compact('categories'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'category_id' => 'required|exists:categories,id',
            'description' => 'required|string',
            'tags' => 'nullable|array',
            'tags.*' => 'string|max:50',
            'images' => 'required|array|min:1',
            'images.*' => 'required|array',
            'images.*.path' => 'required|string',
            'images.*.alt_text' => 'nullable|string|max:255',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'meta_keywords' => 'nullable|string',
            'variants' => 'required|array|min:1',
            'variants.*.unit_type' => 'required|string|max:50',
            'variants.*.unit_value' => 'required|string|max:50',
            'variants.*.sale_price' => 'required|numeric|min:0',
            'variants.*.regular_price' => 'required|numeric|min:0',
            'variants.*.in_stock' => 'required|integer|min:0',
            'variants.*.default' => 'required|boolean',
            'specification' => 'nullable|array',
            'specification.*.key' => 'required|string',
            'specification.*.value' => 'required|string',
            'features' => 'nullable|array|max:5',
            'features.*' => 'required|string'
        ]);

        try {
            DB::beginTransaction();

            // Create product
            $product = Product::create([
                'category_id' => $request->category_id,
                'title' => $request->title,
                'description' => $request->description,
                'brand_id' => $request->brand_id ?: null,
                'tags' => $request->tags,
                'specification' => $request->specification,
                'features' => $request->features,
                'meta_title' => $request->meta_title,
                'meta_description' => $request->meta_description,
                'meta_keywords' => $request->meta_keywords
            ]);

            // Save images
            foreach ($request->images as $index => $imageData) {
                if (isset($imageData['path']) && Str::startsWith($imageData['path'], 'data:image')) {
                    // Handle base64 image
                    $image = base64_decode(preg_replace('#^data:image/\w+;base64,#i', '', $imageData['path']));
                    $filename = 'products/' . $product->id . '/' . uniqid() . '.jpg';
                    Storage::disk('public')->put($filename, $image);
                    
                    $product->images()->create([
                        'path' => $filename,
                        'alt_text' => $imageData['alt_text'] ?? null,
                        'is_primary' => $index === 0,
                        'sort_order' => $index
                    ]);
                }
            }

            // Save variants
            foreach ($request->variants as $variant) {
                $product->variations()->create([
                    'unit_type' => $variant['unit_type'],
                    'unit_value' => $variant['unit_value'],
                    'sale_price' => $variant['sale_price'],
                    'regular_price' => $variant['regular_price'],
                    'in_stock' => $variant['in_stock'],
                    'default' => $variant['default']
                ]);
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Product created successfully'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error creating product: ' . $e->getMessage()
            ], 500);
        }
    }

    public function edit(Product $product)
    {
        $product->load(['category', 'variations', 'images', 'brand']);
        $categories = Category::all();
        return view('admin.products.edit', compact('product', 'categories'));
    }

    public function update(Request $request, Product $product)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'category_id' => 'required|exists:categories,id',
            'description' => 'required|string',
            'tags' => 'nullable|array',
            'tags.*' => 'string|max:50',
            'images' => 'required|array|min:1',
            'images.*' => 'required|array',
            'images.*.path' => 'required|string',
            'images.*.alt_text' => 'nullable|string|max:255',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'meta_keywords' => 'nullable|string',
            'variants' => 'required|array|min:1',
            'variants.*.unit_type' => 'required|string|max:50',
            'variants.*.unit_value' => 'required|string|max:50',
            'variants.*.sale_price' => 'required|numeric|min:0',
            'variants.*.regular_price' => 'required|numeric|min:0',
            'variants.*.in_stock' => 'required|integer|min:0',
            'variants.*.default' => 'required|boolean',
            'specification' => 'nullable|array',
            'specification.*.key' => 'required|string',
            'specification.*.value' => 'required|string',
            'features' => 'nullable|array|max:5',
            'features.*' => 'required|string'
        ]);

        try {
            DB::beginTransaction();

            // Update product
            $product->update([
                'category_id' => $request->category_id,
                'title' => $request->title,
                'description' => $request->description,
                'brand_id' => $request->brand_id ?: null,
                'tags' => $request->tags,
                'specification' => $request->specification,
                'features' => $request->features,
                'meta_title' => $request->meta_title,
                'meta_description' => $request->meta_description,
                'meta_keywords' => $request->meta_keywords
            ]);

            // Handle images
            $existingImages = $product->images->pluck('path')->toArray();
            $newImages = [];
            
            foreach ($request->images as $index => $imageData) {
                if (isset($imageData['path']) && Str::startsWith($imageData['path'], 'data:image')) {
                    // Handle new base64 image
                    $image = base64_decode(preg_replace('#^data:image/\w+;base64,#i', '', $imageData['path']));
                    $filename = 'products/' . $product->id . '/' . uniqid() . '.jpg';
                    Storage::disk('public')->put($filename, $image);
                    $newImages[] = $filename;
                    
                    $product->images()->create([
                        'path' => $filename,
                        'alt_text' => $imageData['alt_text'] ?? null,
                        'is_primary' => $index === 0,
                        'sort_order' => $index
                    ]);
                } else {
                    // Existing image
                    $path = $imageData['path'];
                    $newImages[] = $path;
                    $product->images()
                        ->where('path', $path)
                        ->update([
                            'alt_text' => $imageData['alt_text'] ?? null,
                            'is_primary' => $index === 0, 
                            'sort_order' => $index
                        ]);
                }
            }

            // Remove deleted images
            foreach ($existingImages as $oldImage) {
                if (!in_array($oldImage, $newImages)) {
                    Storage::disk('public')->delete($oldImage);
                }
            }
            $product->images()->whereNotIn('path', $newImages)->delete();

            // Update variants
            $existingVariantIds = $product->variations->pluck('id')->toArray();
            $keepVariantIds = [];

            foreach ($request->variants as $variant) {
                if (isset($variant['id']) && !Str::startsWith($variant['id'], 'new_')) {
                    // Update existing variant
                    $product->variations()
                        ->where('id', $variant['id'])
                        ->update([
                            'unit_type' => $variant['unit_type'],
                            'unit_value' => $variant['unit_value'],
                            'sale_price' => $variant['sale_price'],
                            'regular_price' => $variant['regular_price'],
                            'in_stock' => $variant['in_stock'],
                            'default' => (bool)$variant['default']
                        ]);
                    $keepVariantIds[] = $variant['id'];
                } else {
                    // Create new variant
                    $product->variations()->create([
                        'unit_type' => $variant['unit_type'],
                        'unit_value' => $variant['unit_value'],
                        'sale_price' => $variant['sale_price'],
                        'regular_price' => $variant['regular_price'],
                        'in_stock' => $variant['in_stock'],
                        'default' => (bool)$variant['default']
                    ]);
                }
            }

            // Delete removed variants
            $product->variations()
                ->whereIn('id', array_diff($existingVariantIds, $keepVariantIds))
                ->delete();

            // Ensure exactly one default variant exists
            $hasDefault = $product->variations()->where('default', true)->exists();
            if (!$hasDefault) {
                // If no default variant, set the first one as default
                $product->variations()->first()->update(['default' => true]);
            } else {
                // If multiple defaults, keep only the last one
                $lastDefaultId = collect($request->variants)
                    ->filter(fn($v) => $v['default'])
                    ->last()['id'] ?? null;
                
                if ($lastDefaultId) {
                    // Set all others to false
                    $product->variations()
                        ->where('id', '!=', $lastDefaultId)
                        ->update(['default' => false]);
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Product updated successfully',
                'redirect' => route('admin.products')
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error updating product: ' . $e->getMessage()
            ], 500);
        }
    }

    public function destroy(Product $product)
    {
        try {
            DB::beginTransaction();

            // Delete product images from storage
            foreach ($product->images as $image) {
                Storage::disk('public')->delete($image->path);
            }

            // Delete the product (will cascade delete images and variants)
            $product->delete();

            DB::commit();
            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error deleting product: ' . $e->getMessage()
            ], 500);
        }
    }

    public function reviews(Request $request)
    {
        $query = ProductRating::with([
                'product' => function($q){
                    $q->select('id','title','slug')->withAvg('ratings', 'rating')->withCount('ratings');
                },
                'user:id,first_name,last_name,email'
            ])
            ->when($request->filled('search'), function($q) use ($request) {
                $term = $request->search;
                $q->whereHas('product', function($p) use ($term){
                    $p->where('title', 'like', "%{$term}%");
                })->orWhereHas('user', function($u) use ($term){
                    $u->where('first_name', 'like', "%{$term}%")
                      ->orWhere('last_name', 'like', "%{$term}%")
                      ->orWhere('email', 'like', "%{$term}%");
                });
            });

        $ratings = $query->latest()->paginate(20)->withQueryString();
        return view('admin.products.reviews', compact('ratings'));
    }

    public function destroyReview(ProductRating $rating)
    {
        $rating->delete();
        return redirect()->route('admin.products.reviews')->with('success', 'Review deleted successfully');
    }
} 