<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use Illuminate\Http\Request;
use App\Mail\OrderStatusUpdate;
use App\Mail\PaymentStatus;
use Illuminate\Support\Facades\Mail;
use App\Exports\OrdersExport;
use Maatwebsite\Excel\Facades\Excel;

class OrdersController extends Controller
{
    public function index(Request $request)
    {
        $status = $request->get('status', Order::STATUS_PENDING);
        
        $orders = Order::with(['user', 'items.product', 'items.variation'])
            ->when($status, function($query, $status) {
                return $query->where('order_status', $status);
            })
            ->latest()
            ->paginate(20);

        return view('admin.orders.list', [
            'orders' => $orders,
            'currentStatus' => $status,
            'statuses' => [
                Order::STATUS_PENDING => 'Pending',
                Order::STATUS_PROCESSING => 'Processing',
                Order::STATUS_SHIPPED => 'Shipped',
                Order::STATUS_DELIVERED => 'Delivered',
                Order::STATUS_CANCELLED => 'Cancelled'
            ]
        ]);
    }

    public function show(Order $order)
    {
        $order->load(['user', 'items.product', 'items.variation', 'statuses']);
        
        return view('admin.orders.show', [
            'order' => $order,
            'statuses' => [
                Order::STATUS_PENDING => 'Pending',
                Order::STATUS_PROCESSING => 'Processing',
                Order::STATUS_SHIPPED => 'Shipped',
                Order::STATUS_DELIVERED => 'Delivered',
                Order::STATUS_CANCELLED => 'Cancelled'
            ]
        ]);
    }

    // Legacy manual payment status handler removed

    public function updateStatus(Request $request, Order $order)
    {
        $request->validate([
            'status' => 'required|in:' . implode(',', [
                Order::STATUS_PENDING,
                Order::STATUS_PROCESSING,
                Order::STATUS_SHIPPED,
                Order::STATUS_DELIVERED,
                Order::STATUS_CANCELLED
            ])
        ]);

        $order->addStatus($request->status);

        // Send order status update email
        Mail::to($order->contact_email)
            ->queue(new OrderStatusUpdate($order, $request->status));

        return back()->with('success', 'Order status updated successfully.');
    }

    public function invoice(Order $order)
    {
        $order->load(['items.product', 'items.variation']);

        $orderData = [
            'order_id' => $order->order_number,
            'order_date' => $order->created_at->format('M d, Y H:i'),
            'payment_method' => strtoupper($order->payment_method),
            'payment_status' => $order->payment_status,
            'payment_note' => null,
            'payment_sender' => null,
            'payment_trxid' => $order->payment_trxid,
            'shipping' => $order->getShippingAddress(),
            'cart_items' => $order->items->map(function($item) {
                return [
                    'id' => $item->product_id,
                    'title' => $item->product->title,
                    'image' => $item->product->primary_image->url,
                    'unit' => $item->variation->unit_value . $item->variation->unit_type,
                    'price' => $item->getUnitPrice(),
                    'quantity' => $item->quantity,
                    'total' => $item->getTotal()
                ];
            }),
            'totals' => $order->getTotals(),
            'coupon' => $order->getCouponDetails(),
            'is_in_progress' => $order->isInProgress(),
            'show_success' => false
        ];

        return view('product.order_invoice', compact('orderData'));
    }

    public function generateInvoice(Order $order)
    {
        return view('admin.orders.invoice', compact('order'));
    }

    // Legacy approve/reject flows removed

    // Legacy approve/reject flows removed

    public function export()
    {
        return Excel::download(new OrdersExport, 'orders-' . now()->format('Y-m-d') . '.xlsx');
    }
}