<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Customization;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class CustomizeController extends Controller
{
    public function home()
    {
        $customization = Customization::singleton();
        $home = $customization->home ?? [
            'sliders' => [],
            'services' => [],
            'featured' => [],
            'hero_boxes' => [],
        ];

        return view('admin.customize.home', compact('home'));
    }

    public function homeUpdate(Request $request)
    {
        $customization = Customization::singleton();

        $validated = $request->validate([
            'sliders' => 'array',
            'sliders.*.badge' => 'nullable|string|max:100',
            'sliders.*.title' => 'nullable|string|max:255',
            'sliders.*.slogan' => 'nullable|string|max:255',
            'sliders.*.btn_link' => 'nullable|url|max:255',
            // hero right boxes (3 items)
            'hero_boxes' => 'array',
            'hero_boxes.*.title' => 'nullable|string|max:255',
            'hero_boxes.*.text' => 'nullable|string|max:500',
            'hero_boxes.*.btn_link' => 'nullable|url|max:255',
            'featured' => 'array',
            'featured.*.title' => 'nullable|string|max:255',
            'featured.*.text' => 'nullable|string|max:500',
            'featured.*.btn_link' => 'nullable|url|max:255',
            'services' => 'array',
            'services.*.icon' => 'nullable|string|max:100',
            'services.*.title' => 'nullable|string|max:100',
            'services.*.text' => 'nullable|string|max:255',
        ]);

        $home = [
            'sliders' => [],
            'services' => [],
            'featured' => [],
            'hero_boxes' => [],
        ];

        $existing = $customization->home ?? $home;

        // Handle sliders with optional image uploads
        $sliderInputs = $request->input('sliders', []);
        foreach ($sliderInputs as $index => $item) {
            $slider = [
                'badge' => $item['badge'] ?? null,
                'title' => $item['title'] ?? null,
                'slogan' => $item['slogan'] ?? null,
                'btn_link' => $item['btn_link'] ?? null,
                'bg_img_path' => $existing['sliders'][$index]['bg_img_path'] ?? null,
            ];

            if ($request->hasFile("sliders.$index.bg_img")) {
                // Delete previous file if existed
                if (!empty($slider['bg_img_path'])) {
                    Storage::disk('public')->delete($slider['bg_img_path']);
                }
                $slider['bg_img_path'] = $request->file("sliders.$index.bg_img")->store('site-assets', 'public');
            }

            $home['sliders'][] = $slider;
        }

        // Handle services (no files, simple text/icon)
        $serviceInputs = $request->input('services', []);
        foreach ($serviceInputs as $item) {
            $home['services'][] = [
                'icon' => $item['icon'] ?? null,
                'title' => $item['title'] ?? null,
                'text' => $item['text'] ?? null,
            ];
        }

        // Handle featured items with optional image
        $featuredInputs = $request->input('featured', []);
        foreach ($featuredInputs as $index => $item) {
            $feat = [
                'title' => $item['title'] ?? null,
                'text' => $item['text'] ?? null,
                'btn_link' => $item['btn_link'] ?? null,
                'bg_img_path' => $existing['featured'][$index]['bg_img_path'] ?? null,
            ];

            if ($request->hasFile("featured.$index.bg_img")) {
                if (!empty($feat['bg_img_path'])) {
                    Storage::disk('public')->delete($feat['bg_img_path']);
                }
                $feat['bg_img_path'] = $request->file("featured.$index.bg_img")->store('site-assets', 'public');
            }

            $home['featured'][] = $feat;
        }

        // Handle hero right boxes (3 items, but allow any count provided)
        // Limit to 3 boxes to match frontend layout
        $boxInputs = array_slice($request->input('hero_boxes', []), 0, 3);
        foreach ($boxInputs as $index => $item) {
            $box = [
                'title' => $item['title'] ?? null,
                'text' => $item['text'] ?? null,
                'btn_link' => $item['btn_link'] ?? null,
                'bg_img_path' => $existing['hero_boxes'][$index]['bg_img_path'] ?? null,
            ];

            if ($request->hasFile("hero_boxes.$index.bg_img")) {
                if (!empty($box['bg_img_path'])) {
                    Storage::disk('public')->delete($box['bg_img_path']);
                }
                $box['bg_img_path'] = $request->file("hero_boxes.$index.bg_img")->store('site-assets', 'public');
            }

            $home['hero_boxes'][] = $box;
        }

        $customization->home = $home;
        $customization->save();

        return back()->with('success', 'Home customization updated successfully.');
    }

    

    public function contact()
    {
        $customization = Customization::singleton();
        $contact = $customization->contact ?? [
            'address' => null,
            'phone' => null,
            'email' => null,
            'map_embed' => null
        ];

        return view('admin.customize.contact', compact('contact'));
    }

    public function contactUpdate(Request $request)
    {
        $customization = Customization::singleton();

        $validated = $request->validate([
            'address' => 'nullable|string|max:1000',
            'phone' => 'nullable|string|max:1000',
            'email' => 'nullable|string|max:1000',
            'map_embed' => 'nullable|string|max:2000',
        ]);

        $contact = [
            'address' => $request->input('address'),
            'phone' => $request->input('phone'),
            'email' => $request->input('email'),
            'map_embed' => $request->input('map_embed'),
        ];

        $customization->contact = $contact;
        $customization->save();

        return back()->with('success', 'Contact page customization updated successfully.');
    }
} 