<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class CategoriesController extends Controller
{
    public function index(Request $request)
    {
        if ($request->has('success')) {
            session()->flash('success', $request->query('success'));
        }
        if ($request->has('error')) {
            session()->flash('error', $request->query('error'));
        }

        $categories = Category::with(['parent'])->get();
        $parentCategories = Category::whereNull('parent_id')->get();
        return view('admin.categories.list', compact('categories', 'parentCategories'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'parent_id' => 'nullable|exists:categories,id',
            // Icon is required only for top-level categories
            'icon' => 'nullable|image|mimes:jpeg,png,jpg,svg|max:2048'
        ]);

        try {
            $iconPath = null;
            // Only store icon for top-level categories
            if (!$request->parent_id && $request->hasFile('icon')) {
                $iconPath = $request->file('icon')->store('category-icons', 'public');
            }

            Category::create([
                'name' => $request->name,
                'parent_id' => $request->parent_id,
                'icon' => $iconPath
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Category created successfully'
            ]);
        } catch (\Exception $e) {
            // Delete uploaded file if exists
            if (isset($iconPath)) {
                Storage::disk('public')->delete($iconPath);
            }

            return response()->json([
                'success' => false,
                'message' => 'Error creating category'
            ], 500);
        }
    }

    public function update(Request $request, Category $category)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'parent_id' => 'nullable|exists:categories,id',
            'icon' => 'nullable|image|mimes:jpeg,png,jpg,svg|max:2048'
        ]);

        try {
            $data = ['name' => $request->name];

            // Handle icon for top-level only; ignore for subcategories
            if (!$request->parent_id && $request->hasFile('icon')) {
                if ($category->icon) {
                    Storage::disk('public')->delete($category->icon);
                }
                $data['icon'] = $request->file('icon')->store('category-icons', 'public');
            }

            $data['parent_id'] = $request->parent_id ?: null;

            $category->update($data);

            return response()->json([
                'success' => true,
                'message' => 'Category updated successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating category'
            ], 500);
        }
    }

    public function destroy(Category $category)
    {
        try {
            if ($category->hasChildren()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete category with subcategories'
                ], 422);
            }

            if ($category->products()->exists()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete category with associated products'
                ], 422);
            }

            // Delete icon file
            if ($category->icon) {
                Storage::disk('public')->delete($category->icon);
            }

            $category->delete();
            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting category'
            ], 500);
        }
    }
} 