<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Brand;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class BrandsController extends Controller
{
    public function index(Request $request)
    {
        // Support toast flashes via query params similar to categories page
        if ($request->has('success')) {
            session()->flash('success', $request->query('success'));
        }
        if ($request->has('error')) {
            session()->flash('error', $request->query('error'));
        }

        $brands = Brand::withCount('products')->orderBy('title')->get();
        return view('admin.brands.list', compact('brands'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,webp,svg|max:2048'
        ]);

        $data = [ 'title' => $validated['title'] ];
        if ($request->hasFile('logo')) {
            $data['logo'] = $request->file('logo')->store('brand-logos', 'public');
        }
        Brand::create($data);

        return response()->json(['success' => true, 'message' => 'Brand created successfully']);
    }

    public function update(Request $request, Brand $brand)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,svg|max:2048'
        ]);

        $data = [ 'title' => $validated['title'] ];
        if ($request->hasFile('logo')) {
            if ($brand->logo) {
                Storage::disk('public')->delete($brand->logo);
            }
            $data['logo'] = $request->file('logo')->store('brand-logos', 'public');
        }

        $brand->update($data);
        return response()->json(['success' => true, 'message' => 'Brand updated successfully']);
    }

    public function destroy(Brand $brand)
    {
        if ($brand->logo) {
            Storage::disk('public')->delete($brand->logo);
        }
        $brand->delete();
        return response()->json(['success' => true]);
    }
}


