<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Blogs;
use App\Models\BlogComment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class BlogsController extends Controller
{
    public function index()
    {
        $blogs = Blogs::with(['author', 'comments', 'reactions'])
            ->withCount(['comments', 'reactions'])
            ->latest()
            ->paginate(20);
            
        return view('admin.blogs.list', compact('blogs'));
    }

    public function create()
    {
        return view('admin.blogs.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'thumbnail' => 'required|image|max:2048', // 2MB max
            'tags' => 'nullable|array',
            'tags.*' => 'string|max:50'
        ]);

        try {
            $thumbnail = $request->file('thumbnail');
            $thumbnailPath = $thumbnail->store('blogs/thumbnails', 'public');

            $blog = Blogs::create([
                'user_id' => auth()->id(),
                'title' => $request->title,
                'content' => $request->content,
                'thumbnail' => $thumbnailPath,
                'tags' => $request->tags
            ]);

            session()->flash('success', 'Blog post created successfully.');
            return response()->json([
                'success' => true,
                'message' => 'Blog post created successfully',
                'redirect' => route('admin.blogs')
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error creating blog post: ' . $e->getMessage()
            ], 500);
        }
    }

    public function edit(Blogs $blog)
    {
        return view('admin.blogs.edit', compact('blog'));
    }

    public function update(Request $request, Blogs $blog)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'thumbnail' => 'nullable|image|max:2048', // 2MB max
            'tags' => 'nullable|array',
            'tags.*' => 'string|max:50'
        ]);

        try {
            $data = [
                'title' => $request->title,
                'content' => $request->content,
                'tags' => $request->tags
            ];

            if ($request->hasFile('thumbnail')) {
                // Delete old thumbnail
                if ($blog->thumbnail) {
                    Storage::disk('public')->delete($blog->thumbnail);
                }
                
                // Store new thumbnail
                $thumbnail = $request->file('thumbnail');
                $data['thumbnail'] = $thumbnail->store('blogs/thumbnails', 'public');
            }

            $blog->update($data);

            session()->flash('success', 'Blog post updated successfully.');
            return response()->json([
                'success' => true,
                'message' => 'Blog post updated successfully',
                'redirect' => route('admin.blogs')
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating blog post: ' . $e->getMessage()
            ], 500);
        }
    }

    public function destroy(Blogs $blog)
    {
        try {
            // Delete thumbnail
            if ($blog->thumbnail) {
                Storage::disk('public')->delete($blog->thumbnail);
            }

            // Delete blog (comments and reactions will be cascade deleted)
            $blog->delete();
            session()->flash('success', 'Blog post deleted successfully.');
            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting blog post: ' . $e->getMessage()
            ], 500);
        }
    }

    public function comments()
    {
        // Get all comments (both parent and replies) with relationships
        $allComments = BlogComment::with(['blog', 'user', 'replies.user'])
            ->latest()
            ->get();
            
        // Group comments by blog and organize hierarchically
        $commentsByBlog = $allComments->groupBy('blog_id')->map(function ($blogComments) {
            $parentComments = $blogComments->whereNull('parent_id');
            $replies = $blogComments->whereNotNull('parent_id');
            
            return [
                'blog' => $blogComments->first()->blog,
                'comments' => $parentComments->map(function ($comment) use ($replies) {
                    $commentReplies = $replies->where('parent_id', $comment->id);
                    return [
                        'comment' => $comment,
                        'replies' => $commentReplies
                    ];
                })
            ];
        });
        
        // Flatten the structure to count individual comments + replies
        $flattenedComments = collect();
        foreach ($commentsByBlog as $blogData) {
            foreach ($blogData['comments'] as $commentData) {
                // Add main comment
                $flattenedComments->push([
                    'type' => 'comment',
                    'data' => $commentData,
                    'blog' => $blogData['blog']
                ]);
                
                // Add replies
                foreach ($commentData['replies'] as $reply) {
                    $flattenedComments->push([
                        'type' => 'reply',
                        'data' => $commentData,
                        'reply' => $reply,
                        'blog' => $blogData['blog']
                    ]);
                }
            }
        }
        
        // Paginate the flattened comments
        $perPage = 10;
        $currentPage = request()->get('page', 1);
        
        $paginatedComments = new \Illuminate\Pagination\LengthAwarePaginator(
            $flattenedComments->forPage($currentPage, $perPage),
            $flattenedComments->count(),
            $perPage,
            $currentPage,
            ['path' => request()->url()]
        );
            
        return view('admin.blogs.comments', compact('paginatedComments'));
    }

    public function deleteComment(BlogComment $comment)
    {
        try {
            // This will also delete all replies due to foreign key cascade
            $comment->delete();
            
            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting comment: ' . $e->getMessage()
            ], 500);
        }
    }
} 