<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Order extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'order_number',
        'user_id',
        'contact_name',
        'contact_phone',
        'whatsapp',
        'contact_email',
        'address',
        'city',
        'state',
        'postal_code',
        'subtotal',
        'delivery_charge',
        'discount',
        'total',
        'coupon_code',
        'coupon_type',
        'coupon_value',
        'payment_method',
        'order_status',
        'payment_status',
        'payment_trxid'
    ];

    protected $casts = [
        'subtotal' => 'integer',
        'delivery_charge' => 'integer',
        'discount' => 'integer',
        'total' => 'integer',
        'coupon_value' => 'integer'
    ];

    /**
     * Order statuses
     */
    const STATUS_PENDING = 'pending';
    const STATUS_PROCESSING = 'processing';
    const STATUS_SHIPPED = 'shipped';
    const STATUS_DELIVERED = 'delivered';
    const STATUS_CANCELLED = 'cancelled';

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the order items
     */
    public function items(): HasMany
    {
        return $this->hasMany(OrderItem::class);
    }

    /**
     * Get the payment for this order
     */
    public function payment()
    {
        return $this->hasOne(OrderPayment::class);
    }

    /**
     * Get the user who placed the order
     */
    public function statuses(): HasMany
    {
        return $this->hasMany(OrderStatus::class);
    }

    public function getCurrentStatus(): string
    {
        return $this->order_status;
    }

    public function addStatus(string $status): void
    {
        $this->statuses()->create(['status' => $status]);
        $this->update(['order_status' => $status]);
    }

    public function getShippingAddress(): array
    {
        return [
            'contact_name' => $this->contact_name,
            'contact_phone' => $this->contact_phone,
            'whatsapp' => $this->whatsapp,
            'contact_email' => $this->contact_email,
            'address' => $this->address,
            'city' => $this->city,
            'state' => $this->state,
            'postal_code' => $this->postal_code
        ];
    }

    public function getCouponDetails(): ?array
    {
        if (!$this->coupon_code) {
            return null;
        }

        return [
            'code' => $this->coupon_code,
            'type' => $this->coupon_type,
            'value' => $this->coupon_value,
            'discount' => $this->discount
        ];
    }

    public function getTotals(): array
    {
        return [
            'subtotal' => $this->subtotal,
            'delivery' => $this->delivery_charge,
            'discount' => $this->discount,
            'total' => $this->total
        ];
    }

    public function isInProgress(): bool
    {
        return !in_array($this->order_status, [self::STATUS_DELIVERED, self::STATUS_CANCELLED]);
    }
}