<?php

namespace App\Http\Resources;

use Illuminate\Http\Resources\Json\JsonResource;

class ProductResource extends JsonResource
{
    protected array $toggles = [];

    public function withToggles(array $toggles): self
    {
        $this->toggles = $toggles;
        return $this;
    }

    public function toArray($request): array
    {
        $data = [
            'title' => $this->title,
            'slug' => $this->slug,
            'description' => $this->description,
            'specification' => $this->specification,
            'features' => $this->features,
            'tags' => $this->tags,
            'meta' => [
                'title' => $this->meta_title,
                'description' => $this->meta_description,
                'keywords' => $this->meta_keywords,
            ],
            'ratings' => [
                'average' => $this->average_rating,
                'count' => $this->ratings_count,
            ],
            'updated_at' => optional($this->updated_at)->toIso8601String(),
            'created_at' => optional($this->created_at)->toIso8601String(),
        ];

        if (($this->toggles['category'] ?? true) && $this->relationLoaded('category') && $this->category) {
            // Root category (top-most parent)
            $root = $this->category;
            while ($root && $root->parent) {
                $root = $root->parent;
            }

            // As requested: plain strings only
            $data['category'] = $root ? $root->name : null;

            // Sub category (the product's own category)
            $leaf = $this->category;
            $data['sub_category'] = $leaf ? $leaf->name : null;

            // Keep human readable path for convenience
            $data['category_path'] = $this->category->full_path;
        }

        if (($this->toggles['brand'] ?? true) && $this->relationLoaded('brand') && $this->brand) {
            $data['brand'] = (new BrandResource($this->brand))->toArray($request);
        }

        if (($this->toggles['images'] ?? true) && $this->relationLoaded('images')) {
            $data['images'] = ProductImageResource::collection($this->images)->resolve();
        }

        if (($this->toggles['stocks'] ?? true) && $this->relationLoaded('variations')) {
            $variants = ProductVariationResource::collection($this->variations)->resolve();
            $data['variants'] = $variants;
            // Backward-compatible alias
            $data['stocks'] = $variants;
        }

        if (!($this->toggles['ratings'] ?? true)) {
            unset($data['ratings']);
        }

        return $data;
    }
}