<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use PragmaRX\Google2FA\Google2FA;
use Illuminate\Support\Facades\Session;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Cookie;

class UserController extends Controller
{
    public function account()
    {
        return view('user.account');
    }

    public function updateProfile(Request $request)
    {
        $request->validate([
            'first_name' => ['required', 'string', 'max:255'],
            'last_name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'email', 'max:255', 'unique:users,email,' . auth()->id()],
            'phone' => ['required', 'string', 'max:20']
        ]);

        $user = auth()->user();
        $user->update($request->only(['first_name', 'last_name', 'email', 'phone']));

        return back()->with('success', 'Profile updated successfully!');
    }

    public function shipping()
    {
        return view('user.shipping');
    }

    public function updateShipping(Request $request)
    {
        $data = $request->validate([
            'contact_name' => 'required|string|max:150',
            'contact_phone' => 'required|string|max:100',
            'whatsapp' => 'nullable|string|max:100',
            'contact_email' => 'nullable|email|max:150',
            'address' => 'required|string',
            'city' => 'required|string|max:100',
            'state' => 'required|string|max:150',
            'postal_code' => 'required|string|max:50'
        ]);

        auth()->user()->update_shipping_address($data);

        return back()->with('success', 'Shipping address updated successfully');
    }

    public function wishlist()
    {
        $wishlist = auth()->user()->wishlist()
            ->with(['product.variations' => function($q) {
                $q->where('default', true);
            }, 'product.images' => function($q) {
                $q->where('is_primary', true)
                    ->orWhere(function($sq) {
                        $sq->whereNull('is_primary')
                            ->orderBy('sort_order')
                            ->limit(1);
                    });
            }])
            ->paginate(10);
        return view('user.wishlist', compact('wishlist'));
    }

    public function removeFromWishlist($productId)
    {
        auth()->user()->wishlist()->where('product_id', $productId)->delete();
        return back()->with('success', 'Product removed from wishlist!');
    }

    public function orders()
    {
        $orders = auth()->user()->orders()
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('user.orders', compact('orders'));
    }

    public function orderDetails($orderId)
    {
        // Since we're using the existing invoice page, redirect to it
        return redirect()->route('product.order.invoice', $orderId);
    }

    public function security()
    {
        $user = auth()->user();
        $data = [
            'user' => $user,
            'has2faEnabled' => $user->has_two_factor_enabled()
        ];

        // If 2FA setup is in progress (secret exists but not enabled), regenerate QR code
        if ($user->two_factor_secret && !$user->two_factor_enabled) {
            $google2fa = new Google2FA();
            
            $qrCodeUrl = $google2fa->getQRCodeUrl(
                config('app.name'),
                $user->email,
                $user->two_factor_secret
            );

            // Generate QR code image
            $qrCode = QrCode::size(200)
                           ->format('svg')
                           ->style('round')
                           ->eye('circle')
                           ->backgroundColor(255, 255, 255)
                           ->generate('otpauth://totp/' . urlencode(config('app.name') . ':' . $user->email) . '?secret=' . $user->two_factor_secret . '&issuer=' . urlencode(config('app.name')));

            $data['qrCode'] = $qrCode;
            $data['secret'] = $user->two_factor_secret;
        }

        return view('user.security', $data);
    }

    public function updatePassword(Request $request)
    {
        $request->validate([
            'current_password' => ['required', 'current_password'],
            'password' => ['required', 'confirmed', 'min:6', 'max:128'],
        ]);

        $user = auth()->user();
        $user->password = Hash::make($request->password);
        $user->save();

        return back()->with('success', 'Password updated successfully!');
    }

    public function enable2FA(Request $request)
    {
        $user = auth()->user();
        
        if ($user->has_two_factor_enabled()) {
            return back()->with('error', '2FA is already enabled.');
        }

        $google2fa = new Google2FA();
        $secret = $google2fa->generateSecretKey();
        
        // Generate 8 recovery codes
        $recoveryCodes = collect(range(1, 8))->map(function () {
            return Str::random(10);
        })->toJson();
        
        $user->two_factor_secret = $secret;
        $user->two_factor_recovery_codes = $recoveryCodes;
        $user->save();

        $qrCodeUrl = $google2fa->getQRCodeUrl(
            config('app.name'),
            $user->email,
            $secret
        );

        // Generate QR code image
        $qrCode = QrCode::size(200)
                       ->format('svg')
                       ->style('round')
                       ->eye('circle')
                       ->backgroundColor(255, 255, 255)
                       ->generate('otpauth://totp/' . urlencode(config('app.name') . ':' . $user->email) . '?secret=' . $secret . '&issuer=' . urlencode(config('app.name')));

        return back()->with([
            'success' => '2FA has been enabled. Please scan the QR code with your authenticator app.',
            'qrCode' => $qrCode,
            'secret' => $secret,
            'recoveryCodes' => json_decode($recoveryCodes)
        ]);
    }

    public function confirm2FA(Request $request)
    {
        $request->validate([
            'code' => 'required|string|size:6'
        ]);

        $user = auth()->user();
        $google2fa = new Google2FA();

        // Remove any spaces and ensure uppercase
        $code = str_replace(' ', '', strtoupper($request->code));

        try {
            $valid = $google2fa->verify(
                $code,
                $user->two_factor_secret
            );

            if (!$valid) {
                return back()->with('error', 'Invalid verification code. Please try again.');
            }

            $user->two_factor_enabled = true;
            $user->two_factor_confirmed_at = now();
            
            // Generate recovery codes
            $recoveryCodes = collect(range(1, 8))->map(function () {
                return Str::random(10);
            })->toJson();
            
            $user->two_factor_recovery_codes = $recoveryCodes;
            $user->save();

            // Set 2FA as verified in session
            Session::put('2fa_verified', true);

            // Close the enable2faModal by not passing qrCode in session
            Session::forget('qrCode');
            Session::forget('secret');

            return redirect()->route('security')->with([
                'success' => '2FA has been enabled successfully.',
                'showRecoveryCodes' => true,
                'recoveryCodes' => json_decode($recoveryCodes)
            ]);
        } catch (\Exception $e) {
            return back()->with('error', 'Error verifying code. Please try again.');
        }
    }

    public function disable2FA(Request $request)
    {
        $request->validate([
            'password' => ['required', 'current_password']
        ]);

        $user = auth()->user();
        $user->two_factor_secret = null;
        $user->two_factor_recovery_codes = null;
        $user->two_factor_enabled = false;
        $user->two_factor_confirmed_at = null;
        $user->save();

        // Remove 2FA verification from session
        Session::forget('2fa_verified');
        
        // Remove the remember_2fa cookie
        Cookie::queue(Cookie::forget('remember_2fa'));

        return back()->with('success', '2FA has been disabled successfully.');
    }

    public function cancel2FA()
    {
        $user = auth()->user();
        
        // Only allow canceling if 2FA is not fully enabled
        if ($user->two_factor_enabled) {
            return back()->with('error', 'Cannot cancel 2FA as it is already enabled. Use disable instead.');
        }

        // Reset 2FA fields
        $user->two_factor_secret = null;
        $user->two_factor_recovery_codes = null;
        $user->two_factor_confirmed_at = null;
        $user->save();

        return back()->with('success', '2FA setup has been cancelled.');
    }

    public function show2FAVerify()
    {
        if (!auth()->user()->has_two_factor_enabled()) {
            return redirect()->route('account');
        }
        
        return view('auth.verify-2fa');
    }

    public function verify2FA(Request $request)
    {
        $request->validate([
            'code' => 'required|string|max:10'
        ]);

        $user = auth()->user();
        $google2fa = new Google2FA();

        // Check if it's a recovery code
        if (strlen($request->code) === 10) {
            $recoveryCodes = json_decode($user->two_factor_recovery_codes, true);
            
            if (in_array($request->code, $recoveryCodes)) {
                // Remove used recovery code
                $recoveryCodes = array_diff($recoveryCodes, [$request->code]);
                $user->two_factor_recovery_codes = json_encode(array_values($recoveryCodes));
                $user->save();
                
                Session::put('2fa_verified', true);
                Cookie::queue('remember_2fa', true, 43200); // 30 days (60*24*30)
                return redirect()->intended();
            }
            
            return back()->with('error', 'Invalid recovery code.');
        }

        // Regular 2FA code verification
        $code = str_replace(' ', '', strtoupper($request->code));

        try {
            $valid = $google2fa->verify(
                $code,
                $user->two_factor_secret
            );

            if (!$valid) {
                return back()->with('error', 'Invalid verification code. Please try again.');
            }

            Session::put('2fa_verified', true);
            Cookie::queue('remember_2fa', true, 43200); // 30 days (60*24*30)
            return redirect()->intended();
        } catch (\Exception $e) {
            return back()->with('error', 'Error verifying code. Please try again.');
        }
    }

    public function regenerateRecoveryCodes(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->has_two_factor_enabled()) {
            return back()->with('error', '2FA is not enabled.');
        }

        // Generate 8 new recovery codes
        $recoveryCodes = collect(range(1, 8))->map(function () {
            return Str::random(10);
        })->toJson();
        
        $user->two_factor_recovery_codes = $recoveryCodes;
        $user->save();

        return back()->with([
            'success' => 'Recovery codes have been regenerated.',
            'recoveryCodes' => json_decode($recoveryCodes)
        ]);
    }

    public function deleteAccount(Request $request)
    {
        $request->validate([
            'password' => ['required', 'current_password']
        ]);

        $user = auth()->user();
        auth()->logout();
        $user->delete();

        return redirect()->route('home')->with('success', 'Your account has been permanently deleted.');
    }
}