<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use App\Models\Brand;
use Illuminate\Support\Str;
use App\Models\Category;
use App\Models\Product;
use App\Models\ProductImage;
use App\Models\ProductVariation;
use App\Models\Coupons;
use App\Models\Order;
// Legacy manual payment artifacts removed
use App\Mail\OrderConfirmation;
use App\Mail\Admin\NewOrderNotification;
use App\Mail\Admin\LowStockNotification;

class ProductController extends Controller
{
    // Define constants for sorting options
    const SORT_OPTIONS = [
        'latest' => ['created_at', 'desc'],
        'oldest' => ['created_at', 'asc'],
        'price_low' => ['sale_price', 'asc'],
        'price_high' => ['sale_price', 'desc'],
        'name_asc' => ['title', 'asc'],
        'name_desc' => ['title', 'desc'],
    ];

    // Reviews pagination size (change here for testing)
    const REVIEWS_PER_PAGE = 20;

    public function cart()
    {
        return view('product.cart');
    }
    
    public function checkout(Request $request)
    {
        if (!auth()->check()) {
            return redirect()->route('login')->with('error', 'Please login to proceed to checkout.');
        }
        if (!auth()->user()->is_active || is_null(auth()->user()->email_verified_at)) {
            return redirect()->back()->with('error', 'Your account must be active and email verified to proceed to checkout.');
        }

        // Only COD and SSLCommerz remain
        return view('product.checkout');
    }

    public function list(Request $request)
    {
        // Get all root categories with their children (no departments)
        $categories = Category::select('id', 'name', 'slug', 'parent_id')
            ->with(['children:id,name,slug,parent_id'])
            ->whereNull('parent_id')
            ->orderBy('name')
            ->get();

        // Initialize query builder
        $query = Product::with(['category', 'brand', 'variations' => function($q) {
            $q->where('default', true);
        }, 'images' => function($q) {
            $q->where('is_primary', true);
        }])
        ->withCount([
            'variations as variants_count' => function($q) {},
            'variations as in_stock_variants_count' => function($q) {
                $q->where('in_stock', '>', 0);
            }
        ])
        ->select('products.*');

        // Apply filters
        $this->applyFilters($query, $request);

        // Get products with pagination
        $products = $query->paginate(24)->withQueryString();

        // Get active filters for the view
        $activeFilters = $this->getActiveFilters($request);

        $brands = Brand::orderBy('title')->get();
        return view('product.list', compact('categories', 'products', 'activeFilters', 'brands'));
    }

    /**
     * Apply filters to the query based on request parameters
     */
    private function applyFilters(Builder $query, Request $request): void
    {
        // Category filter
        if ($request->filled('category')) {
            $query->whereHas('category', function ($q) use ($request) {
                $q->where('slug', $request->category)
                  ->orWhereHas('parent', function($q) use ($request) {
                      $q->where('slug', $request->category);
                  });
            });
        }

        // Department filter removed

        // Brand filter (accepts id or brand name/slug)
        if ($request->filled('brand')) {
            $brandParam = (string) $request->brand;
            $query->where(function($q) use ($brandParam) {
                if (ctype_digit($brandParam)) {
                    $q->where('brand_id', $brandParam);
                } else {
                    $slug = Str::slug($brandParam);
                    $q->whereHas('brand', function($b) use ($brandParam, $slug) {
                        $b->whereRaw('LOWER(title) = ?', [strtolower($brandParam)])
                          ->orWhereRaw('LOWER(REPLACE(title, " ", "-")) = ?', [$slug]);
                    });
                }
            });
        }

        // Price range filter
        if ($request->filled('price_range')) {
            [$min, $max] = explode('-', $request->price_range);
            $query->whereHas('variations', function ($q) use ($min, $max) {
                $q->where('sale_price', '>=', $min)
                  ->where('sale_price', '<=', $max);
            });
        }

        // Size filter
        if ($request->filled('sizes')) {
            $sizes = is_array($request->sizes) ? $request->sizes : [$request->sizes];
            $query->whereHas('variations', function ($q) use ($sizes) {
                $q->whereIn('unit_value', $sizes);
            });
        }

        // Color filter
        if ($request->filled('colors')) {
            $colors = explode(',', $request->colors);
            $query->whereHas('variations', function ($q) use ($colors) {
                $q->where('unit_type', 'color')
                  ->whereIn('unit_value', $colors);
            });
        }

        // Search term
        if ($request->filled('search')) {
            $searchTerm = $request->search;
            $query->where(function ($q) use ($searchTerm) {
                $q->where('title', 'like', "%{$searchTerm}%")
                  ->orWhere('description', 'like', "%{$searchTerm}%")
                  ->orWhereHas('category', function ($q) use ($searchTerm) {
                      $q->where('name', 'like', "%{$searchTerm}%");
                  });
            });
        }

        // Sorting
        $sortBy = $request->input('sort', 'latest');
        if (isset(self::SORT_OPTIONS[$sortBy])) {
            [$column, $direction] = self::SORT_OPTIONS[$sortBy];
            
            // Special handling for price sorting since it's in variations table
            if (in_array($sortBy, ['price_low', 'price_high'])) {
                $query->join('product_variations', function($join) {
                    $join->on('products.id', '=', 'product_variations.product_id')
                         ->where('product_variations.default', '=', true);
                })
                ->orderBy('product_variations.sale_price', $direction);
            } else {
                $query->orderBy($column, $direction);
            }
        }
    }

    /**
     * Get active filters for displaying in the view
     */
    private function getActiveFilters(Request $request): array
    {
        $activeFilters = [];

        if ($request->filled('category')) {
            $category = Category::where('slug', $request->category)->first();
            if ($category) {
                $activeFilters['category'] = $category->name;
            }
        }

        // Brand label for chip (supports id or slug/name)
        if ($request->filled('brand')) {
            $brandParam = (string) $request->brand;
            $brand = null;
            if (ctype_digit($brandParam)) {
                $brand = Brand::find($brandParam);
            } else {
                $slug = Str::slug($brandParam);
                $brand = Brand::whereRaw('LOWER(title) = ?', [strtolower($brandParam)])
                    ->orWhereRaw('LOWER(REPLACE(title, " ", "-")) = ?', [$slug])
                    ->first();
            }
            $activeFilters['brand'] = $brand?->title ?? ucwords(str_replace('-', ' ', $brandParam));
        }


        if ($request->filled(['price_min', 'price_max'])) {
            $activeFilters['price_range'] = "৳{$request->price_min} - ৳{$request->price_max}";
        }

        if ($request->filled('sizes')) {
            $activeFilters['sizes'] = is_array($request->sizes) ? implode(', ', $request->sizes) : $request->sizes;
        }

        if ($request->filled('colors')) {
            $activeFilters['colors'] = is_array($request->colors) ? implode(', ', $request->colors) : $request->colors;
        }

        if ($request->filled('search')) {
            $activeFilters['search'] = $request->search;
        }

        return $activeFilters;
    }

    public function view($slug)
    {
        // Find the product with its relationships
        $product = Product::with([
            
            'category',
            'variations',
            'images',
            'category.parent',
            'ratings.user'
        ])->where('slug', $slug)->firstOrFail();

        // Get related products based on category hierarchy
        $relatedProductsQuery = Product::with(['category', 'variations' => function($q) {
            $q->where('default', true);
        }, 'images' => function($q) {
            $q->where('is_primary', true);
        }])
        ->withCount([
            'variations as variants_count' => function($q) {},
            'variations as in_stock_variants_count' => function($q) {
                $q->where('in_stock', '>', 0);
            }
        ])
        ->where('id', '!=', $product->id);

        // If product is in a sub-category (has parent), get products from same sub-category
        // Otherwise, get products from same main category
        if ($product->category->parent_id) {
            // Product is in a sub-category, get products from same sub-category
            $relatedProductsQuery->where('category_id', $product->category_id);
        } else {
            // Product is in a main category, get products from same main category or its sub-categories
            $relatedProductsQuery->whereHas('category', function($query) use ($product) {
                $query->where('id', $product->category_id)
                      ->orWhere('parent_id', $product->category_id);
            });
        }

        $relatedProducts = $relatedProductsQuery->take(4)->get();

        

        // Add stock threshold constants
        $stockThresholds = [
            'critical' => ProductVariation::CRITICAL_STOCK_THRESHOLD,
            'low' => ProductVariation::LOW_STOCK_THRESHOLD
        ];

        // Ratings summary and reviews
        $totalRatings = $product->ratings()->count();
        $averageRating = $totalRatings ? round((float)$product->ratings()->avg('rating'), 2) : 0.0;
        $ratingCounts = $product->ratings()
            ->selectRaw('rating, COUNT(*) as count')
            ->groupBy('rating')
            ->pluck('count', 'rating');
        $distribution = [];
        for ($i = 5; $i >= 1; $i--) {
            $distribution[$i] = (int)($ratingCounts[$i] ?? 0);
        }

        // Reviews sorting and filtering
        $sort = request('sort', 'latest'); // latest|oldest|highest|lowest
        $star = (int) request('star', 0);  // 0 = all, 1..5

        $reviewsQuery = $product->ratings()->with('user');
        if ($star >= 1 && $star <= 5) {
            $reviewsQuery->where('rating', $star);
        }
        switch ($sort) {
            case 'oldest':
                $reviewsQuery->oldest();
                break;
            case 'highest':
                $reviewsQuery->orderByDesc('rating')->orderByDesc('created_at');
                break;
            case 'lowest':
                $reviewsQuery->orderBy('rating')->orderByDesc('created_at');
                break;
            default:
                $reviewsQuery->latest();
        }
        $reviews = $reviewsQuery->paginate(self::REVIEWS_PER_PAGE)->withQueryString();

        $canReview = false;
        if (auth()->check()) {
            $hasPurchased = Order::where('user_id', auth()->id())
                ->where(function($q){
                    $q->where('order_status', Order::STATUS_DELIVERED)
                      ->orWhere('order_status', Order::STATUS_SHIPPED)
                      ->orWhere('order_status', Order::STATUS_PROCESSING);
                })
                ->whereHas('items', function($q) use ($product) {
                    $q->where('product_id', $product->id);
                })
                ->exists();

            $alreadyReviewed = $product->ratings()->where('user_id', auth()->id())->exists();
            $canReview = $hasPurchased && !$alreadyReviewed;
        }

        return view('product.view', compact(
            'product',
            'relatedProducts',
            'stockThresholds',
            'totalRatings',
            'averageRating',
            'distribution',
            'reviews',
            'canReview',
            'sort',
            'star'
        ));
    }

    public function storeRating(Request $request, Product $product)
    {
        $this->validate($request, [
            'rating' => 'required|integer|min:1|max:5',
            'comment' => 'nullable|string|max:2000',
        ]);

        $userId = auth()->id();

        // Check purchase eligibility
        $hasPurchased = Order::where('user_id', $userId)
            ->where(function($q){
                $q->where('order_status', Order::STATUS_DELIVERED)
                  ->orWhere('order_status', Order::STATUS_SHIPPED)
                  ->orWhere('order_status', Order::STATUS_PROCESSING);
            })
            ->whereHas('items', function($q) use ($product) {
                $q->where('product_id', $product->id);
            })
            ->exists();

        if (!$hasPurchased) {
            if ($request->ajax()) {
                return response()->json(['success' => false, 'message' => 'Only customers who purchased this product can submit a review.'], 403);
            }
            return redirect()->back()->with('error', 'Only customers who purchased this product can submit a review.');
        }

        // Prevent duplicate reviews per user per product
        if ($product->ratings()->where('user_id', $userId)->exists()) {
            if ($request->ajax()) {
                return response()->json(['success' => false, 'message' => 'You have already reviewed this product.'], 422);
            }
            return redirect()->back()->with('error', 'You have already reviewed this product.');
        }

        $product->ratings()->create([
            'user_id' => $userId,
            'rating' => (int)$request->input('rating'),
            'comment' => $request->input('comment'),
        ]);

        if ($request->ajax()) {
            // Return refreshed partial for reviews block
            $html = $this->renderReviewsPartial($product);
            return response()->json(['success' => true, 'html' => $html]);
        }
        return redirect(route('product.view', ['slug' => $product->slug]) . '#mn-spt-nav-review')
            ->with('success', 'Thanks for your review!');
    }

    public function reviews(Request $request, Product $product)
    {
        return $this->renderReviewsPartial($product);
    }

    private function renderReviewsPartial(Product $product)
    {
        // recompute summary, distribution, and filtered/sorted reviews
        $totalRatings = $product->ratings()->count();
        $averageRating = $totalRatings ? round((float)$product->ratings()->avg('rating'), 2) : 0.0;
        $ratingCounts = $product->ratings()
            ->selectRaw('rating, COUNT(*) as count')
            ->groupBy('rating')
            ->pluck('count', 'rating');
        $distribution = [];
        for ($i = 5; $i >= 1; $i--) {
            $distribution[$i] = (int)($ratingCounts[$i] ?? 0);
        }

        $sort = request('sort', 'latest');
        $star = (int) request('star', 0);
        $reviewsQuery = $product->ratings()->with('user');
        if ($star >= 1 && $star <= 5) {
            $reviewsQuery->where('rating', $star);
        }
        switch ($sort) {
            case 'oldest': $reviewsQuery->oldest(); break;
            case 'highest': $reviewsQuery->orderByDesc('rating')->orderByDesc('created_at'); break;
            case 'lowest': $reviewsQuery->orderBy('rating')->orderByDesc('created_at'); break;
            default: $reviewsQuery->latest();
        }
        $reviews = $reviewsQuery->paginate(self::REVIEWS_PER_PAGE)->withQueryString();

        $canReview = false;
        if (auth()->check()) {
            $hasPurchased = Order::where('user_id', auth()->id())
                ->where(function($q){
                    $q->where('order_status', Order::STATUS_DELIVERED)
                      ->orWhere('order_status', Order::STATUS_SHIPPED)
                      ->orWhere('order_status', Order::STATUS_PROCESSING);
                })
                ->whereHas('items', function($q) use ($product) { $q->where('product_id', $product->id); })
                ->exists();
            $alreadyReviewed = $product->ratings()->where('user_id', auth()->id())->exists();
            $canReview = $hasPurchased && !$alreadyReviewed;
        }

        return view('partial.reviews', compact(
            'product','totalRatings','averageRating','distribution','reviews','canReview','sort','star'
        ))->render();
    }

    public function destroyRating(Request $request, Product $product)
    {
        if (!auth()->check()) {
            if ($request->ajax()) {
                return response()->json(['success' => false, 'message' => 'Unauthorized'], 401);
            }
            abort(401);
        }
        $deleted = $product->ratings()->where('user_id', auth()->id())->delete();
        if ($request->ajax()) {
            $html = $this->renderReviewsPartial($product);
            return response()->json(['success' => true, 'html' => $html]);
        }
        return redirect()->back();
    }

    /**
     * Toggle product in user's wishlist
     */
    public function toggleWishlist(Product $product)
    {
        if (!auth()->check()) {
            return response()->json([
                'success' => false,
                'message' => 'Please login to add items to wishlist',
                'require_login' => true
            ], 401);
        }

        $user = auth()->user();
        $exists = $user->wishlist()->where('product_id', $product->id)->exists();

        if ($exists) {
            $user->wishlist()->where('product_id', $product->id)->delete();
            return response()->json([
                'success' => true,
                'in_wishlist' => false,
                'message' => 'Product removed from wishlist'
            ]);
        }

        $user->wishlist()->create(['product_id' => $product->id]);
        return response()->json([
            'success' => true,
            'in_wishlist' => true,
            'message' => 'Product added to wishlist'
        ]);
    }

    /**
     * Check if product is in user's wishlist
     */
    public function checkWishlist(Product $product)
    {
        if (!auth()->check()) {
            return response()->json(['in_wishlist' => false]);
        }

        $inWishlist = auth()->user()->wishlist()
            ->where('product_id', $product->id)
            ->exists();

        return response()->json(['in_wishlist' => $inWishlist]);
    }

    /**
     * Validate coupon code and return discount info
     */
    public function validateCoupon(Request $request)
    {
        $request->validate([
            'code' => 'required|string',
            'subtotal' => 'required|numeric|min:0'
        ]);

        // Using whereRaw with BINARY for case-sensitive comparison
        $coupon = Coupons::whereRaw('BINARY code = ?', [$request->code])
            ->where('valid_from', '<=', now())
            ->where('valid_until', '>=', now())
            ->first();

        if (!$coupon) {
            return response()->json([
                'valid' => false,
                'message' => 'Invalid or expired coupon code!'
            ], 422);
        }

        // Check usage limit
        if ($coupon->usage_limit && $coupon->usage_count >= $coupon->usage_limit) {
            return response()->json([
                'valid' => false,
                'message' => 'Coupon usage limit reached'
            ], 422);
        }

        // Calculate discount
        $discount = 0;
        if ($coupon->type === 'fixed') {
            $discount = $coupon->value;
        } else {
            $discount = round(($request->subtotal * $coupon->value) / 100);
        }

        return response()->json([
            'valid' => true,
            'discount' => $discount,
            'type' => $coupon->type,
            'value' => $coupon->value,
            'message' => 'Coupon applied successfully'
        ]);
    }

    public function placeOrder(Request $request)
    {
        // Validate the request
        $request->validate([
            'contact_name' => 'required|string|max:255',
            'contact_phone' => 'required|string|max:20',
            'whatsapp' => 'nullable|string|max:20',
            'contact_email' => 'nullable|email|max:255',
            'address' => 'required|string',
            'city' => 'required|string|max:100',
            'state' => 'required|string|max:100',
            'postal_code' => 'required|string|max:20',
            'payment_method' => 'required|in:cod,sslcommerz',
            'coupon_code' => 'nullable|string',
            'cart_items' => 'required|json'
        ]);

        try {
            DB::beginTransaction();

            // No manual payment methods anymore

            // Get cart items and fetch their details from database
            $cartItems = json_decode($request->cart_items, true);
            
            // Fetch products with their variations
            $products = Product::with(['variations'])
                ->whereIn('id', array_column($cartItems, 'productId'))
                ->get();

            $processedItems = [];
            $subtotal = 0;
            $stockError = false;

            // Process cart items and calculate totals
            foreach ($cartItems as $cartItem) {
                $product = $products->find($cartItem['productId']);
                if (!$product) continue;

                $variation = $product->variations->find($cartItem['variantId']);
                if (!$variation) continue;

                // Check if enough stock is available
                if ($variation->in_stock < $cartItem['quantity']) {
                    $stockError = true;
                    return response()->json([
                        'success' => false,
                        'message' => "Sorry! Only {$variation->in_stock} items available for {$product->title} ({$variation->unit_value} {$variation->unit_type})"
                    ], 422);
                }

                $itemTotal = $variation->sale_price * $cartItem['quantity'];
                $subtotal += $itemTotal;

                $processedItems[] = [
                    'product_id' => $product->id,
                    'variation_id' => $variation->id,
                    'quantity' => $cartItem['quantity']
                ];
            }

            // Process coupon if applied
            $couponDiscount = 0;
            $couponInfo = null;
            if ($request->filled('coupon_code')) {
                $couponData = json_decode($request->coupon_code, true);
                if ($couponData) {
                    // Validate coupon again to prevent tampering
                    $coupon = Coupons::whereRaw('BINARY code = ?', [$couponData['code']])
                        ->where('valid_from', '<=', now())
                        ->where('valid_until', '>=', now())
                        ->first();

                    if (!$coupon) {
                        return response()->json([
                            'success' => false,
                            'message' => 'Invalid or expired coupon code!'
                        ], 422);
                    }

                    // Check usage limit
                    if ($coupon->usage_limit && $coupon->usage_count >= $coupon->usage_limit) {
                        return response()->json([
                            'success' => false,
                            'message' => 'Coupon usage limit reached!'
                        ], 422);
                    }

                    $couponInfo = [
                        'code' => $couponData['code'],
                        'type' => $couponData['type'],
                        'value' => (int)$couponData['value'],
                        'discount' => (int)$couponData['discount']
                    ];
                    $couponDiscount = $couponInfo['discount'];

                    // Update coupon usage count
                    $coupon->increment('usage_count');
                }
            }

            // Calculate final totals
            $deliveryCharge = 60; // You might want to make this configurable
            $total = $subtotal + $deliveryCharge - $couponDiscount;

            // Create the order
            $order = Order::create([
                'order_number' => 'SB' . strtoupper(uniqid()),
                'user_id' => auth()->id(),
                'contact_name' => $request->contact_name,
                'contact_phone' => $request->contact_phone,
                'whatsapp' => $request->whatsapp,
                'contact_email' => $request->contact_email,
                'address' => $request->address,
                'city' => $request->city,
                'state' => $request->state,
                'postal_code' => $request->postal_code,
                'subtotal' => $subtotal,
                'delivery_charge' => $deliveryCharge,
                'discount' => $couponDiscount,
                'total' => $total,
                'coupon_code' => $couponInfo ? $couponInfo['code'] : null,
                'coupon_type' => $couponInfo ? $couponInfo['type'] : null,
                'coupon_value' => $couponInfo ? $couponInfo['value'] : null,
                'payment_method' => $request->payment_method,
                'order_status' => Order::STATUS_PENDING
            ]);

            // Create order items and update stock
            foreach ($processedItems as $item) {
                $order->items()->create([
                    'product_id' => $item['product_id'],
                    'product_variation_id' => $item['variation_id'],
                    'quantity' => $item['quantity']
                ]);

                // Decrease stock and check for low stock
                $variation = ProductVariation::find($item['variation_id']);
                $variation->decrement('in_stock', $item['quantity']);
                $variation->refresh(); // Get the updated stock count

                // If stock is low, get the product data properly
                if ($variation->in_stock <= ProductVariation::LOW_STOCK_THRESHOLD) {
                    // Get the product first
                    $product = Product::find($variation->product_id);
                    
                    if ($product) {
                        $lowStockItems = [[
                            'product' => $product,
                            'variation' => $variation,
                            'stock' => $variation->in_stock
                        ]];
                        
                        Mail::to(env('ADMIN_NOTIFIER_MAIL'))
                            ->queue(new LowStockNotification($lowStockItems));
                    }
                }
            }

            // Create initial order status
            $order->addStatus(Order::STATUS_PENDING);

            // If SSLCommerz selected, init gateway and return redirect URL
            if ($request->payment_method === 'sslcommerz') {
                $order->update(['payment_status' => 'unpaid']);

                $tranId = $order->order_number;
                $postData = [
                    'store_id' => config('services.sslcommerz.store_id'),
                    'store_passwd' => config('services.sslcommerz.store_password'),
                    'total_amount' => $total,
                    'currency' => 'BDT',
                    'tran_id' => $tranId,
                    'success_url' => route('payment.ssl.success'),
                    'fail_url' => route('payment.ssl.fail'),
                    'cancel_url' => route('payment.ssl.cancel'),
                    'ipn_url' => route('payment.ssl.ipn'),
                    'cus_name' => $request->contact_name,
                    'cus_email' => $request->contact_email ?? 'noemail@example.com',
                    'cus_add1' => $request->address,
                    'cus_city' => $request->city,
                    'cus_state' => $request->state,
                    'cus_postcode' => $request->postal_code,
                    'cus_country' => 'Bangladesh',
                    'cus_phone' => $request->contact_phone,
                    'shipping_method' => 'NO',
                    'product_name' => 'Order ' . $tranId,
                    'product_category' => 'Mixed',
                    'product_profile' => 'general'
                ];

                $isSandbox = (bool) config('services.sslcommerz.sandbox', true);
                $gatewayUrl = $isSandbox
                    ? 'https://sandbox.sslcommerz.com/gwprocess/v4/api.php'
                    : 'https://securepay.sslcommerz.com/gwprocess/v4/api.php';

                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $gatewayUrl);
                curl_setopt($ch, CURLOPT_POST, 1);
                curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postData));
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                $response = curl_exec($ch);
                $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                curl_close($ch);

                $gw = json_decode($response, true);
                if ($httpCode === 200 && isset($gw['status']) && $gw['status'] === 'SUCCESS' && !empty($gw['GatewayPageURL'])) {
                    DB::commit();
                    return response()->json([
                        'success' => true,
                        'redirect_to_gateway' => $gw['GatewayPageURL']
                    ]);
                }

                throw new \RuntimeException('Failed to initiate SSLCommerz payment');
            }

            // Queue all emails within the transaction
            if ($order->contact_email) {
                Mail::to($order->contact_email)
                    ->queue(new OrderConfirmation($order));
            }

            Mail::to(env('ADMIN_NOTIFIER_MAIL'))
                ->queue(new NewOrderNotification($order));

            // No manual payment email anymore

            DB::commit();

            // Flash success message that will be available only for the next request
            session()->flash('order_success', true);

            // Return success response (COD)
            return response()->json([
                'success' => true,
                'redirect_url' => route('product.order.invoice', ['order_id' => $order->order_number])
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Order placement failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to place order. Please try again.'
            ], 500);
        }
    }

    // SSLCommerz callbacks
    public function sslSuccess(Request $request)
    {
        // Handle both GET and POST requests from SSLCommerz
        $tranId = $request->input('tran_id');
        $valId = $request->input('val_id');
        $bankTranId = $request->input('bank_tran_id');
        
        Log::info('SSLCommerz success callback received', [
            'tran_id' => $tranId,
            'val_id' => $valId,
            'all_params' => $request->all()
        ]);
        
        if (!$tranId) {
            Log::error('SSLCommerz success callback missing transaction ID');
            return redirect()->route('product.checkout')
                ->with('error', 'Invalid transaction data. Please try again.');
        }
        
        try {
            $order = Order::where('order_number', $tranId)->firstOrFail();

            // Validate payment with SSLCommerz if val_id present
            $validated = false;
            $trxIdToSave = $bankTranId ?: $valId ?: $tranId;
            if ($valId) {
                try {
                    $validatorUrl = (bool) config('services.sslcommerz.sandbox', true)
                        ? 'https://sandbox.sslcommerz.com/validator/api/validationserverAPI.php'
                        : 'https://securepay.sslcommerz.com/validator/api/validationserverAPI.php';
                    $query = http_build_query([
                        'val_id' => $valId,
                        'store_id' => config('services.sslcommerz.store_id'),
                        'store_passwd' => config('services.sslcommerz.store_password'),
                        'format' => 'json'
                    ]);
                    $resp = @file_get_contents($validatorUrl.'?'.$query);
                    if ($resp !== false) {
                        $data = json_decode($resp, true);
                        Log::info('SSLCommerz validation response', ['data' => $data]);
                        if (!empty($data['status']) && in_array($data['status'], ['VALID', 'VALIDATED'])) {
                            $validated = true;
                            // prefer bank_tran_id from validator
                            if (!empty($data['bank_tran_id'])) {
                                $trxIdToSave = $data['bank_tran_id'];
                            }
                        }
                    }
                } catch (\Throwable $t) {
                    Log::warning('SSLCommerz validation call failed', ['error' => $t->getMessage()]);
                }
            }

            // If not validated via API, trust IPN/success combo when status flag provided
            if (!$validated && strtoupper((string)$request->input('status')) === 'VALID') {
                $validated = true;
            }

            if ($validated) {
                // Update payment status and transaction ID
                $order->update([
                    'payment_status' => 'paid',
                    'payment_trxid' => $trxIdToSave
                ]);

                // Do NOT advance order status automatically.
                // Keep it at 'pending' until an admin processes the order.

                // Send order confirmation email if not already sent
                try {
                    if ($order->contact_email) {
                        Mail::to($order->contact_email)
                            ->queue(new OrderConfirmation($order));
                    }
                } catch (\Exception $e) {
                    Log::error('Failed to send order confirmation email: ' . $e->getMessage());
                }

                // Flash a flag to clear the cart on the frontend
                session()->flash('clear_cart', true);
                session()->flash('order_success', true);

                return redirect()->route('product.order.invoice', ['order_id' => $order->order_number])
                    ->with('success', 'Payment successful! Your order has been placed.');
            }

            // Fallback: if not validated, show invoice but keep unpaid status
            return redirect()->route('product.order.invoice', ['order_id' => $order->order_number])
                ->with('info', 'Payment processing. We will update the status shortly.');
        } catch (\Exception $e) {
            Log::error('SSLCommerz success callback processing error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return redirect()->route('product.checkout')
                ->with('error', 'There was a problem processing your payment. Please contact support.');
        }
    }

    public function sslFail(Request $request)
    {
        $tranId = $request->input('tran_id');
        
        // Handle case when tran_id is missing
        if (!$tranId) {
            return redirect()->route('product.checkout')
                ->with('error', 'Payment failed. Please try again.');
        }
        
        $order = Order::where('order_number', $tranId)->first();
        if ($order) {
            $order->update(['payment_status' => 'failed']);
            // Add a status entry for the failed payment
            $order->addStatus(Order::STATUS_PENDING, 'Payment failed');
            
            // Redirect to the invoice page so they can try again
            return redirect()->route('product.order.invoice', ['order_id' => $order->order_number])
                ->with('error', 'Payment failed. You can try again or choose a different payment method.')
                ->with('payment_failed', true);
        }
        
        return redirect()->route('product.checkout')
            ->with('error', 'Payment failed. Please try again.');
    }

    public function sslCancel(Request $request)
    {
        $tranId = $request->input('tran_id');
        
        // Handle case when tran_id is missing
        if (!$tranId) {
            return redirect()->route('product.checkout')
                ->with('error', 'Payment was cancelled.');
        }
        
        $order = Order::where('order_number', $tranId)->first();
        if ($order) {
            $order->update(['payment_status' => 'cancelled']);
            
            // Redirect to the invoice page so they can try again
            return redirect()->route('product.order.invoice', ['order_id' => $order->order_number])
                ->with('info', 'Payment was cancelled. You can try again or choose a different payment method.')
                ->with('payment_failed', true);
        }
        
        return redirect()->route('product.checkout')
            ->with('error', 'Payment was cancelled.');
    }

    public function sslIpn(Request $request)
    {
        // Minimal IPN: mark paid if status is VALID
        $tranId = $request->input('tran_id');
        $status = $request->input('status');
        if ($tranId && $status === 'VALID') {
            $order = Order::where('order_number', $tranId)->first();
            if ($order && $order->payment_status !== 'paid') {
                $order->update(['payment_status' => 'paid', 'payment_trxid' => $request->input('val_id')]);
            }
        }
        return response('OK');
    }

    public function orderInvoice($order_id)
    {
        // Find the order by order number
        $order = Order::with(['items.product', 'items.variation'])
            ->where('order_number', $order_id)
            ->firstOrFail();

        // Prepare order data for the view
        $orderData = [
            'order_id' => $order->order_number,
            'order_date' => $order->created_at->format('d M, Y h:i A'),
            'payment_method' => $order->payment_method,
            'payment_method_name' => strtoupper($order->payment_method) === 'COD' ? 'Cash On Delivery' : 'SSLCommerz',
            'payment_status' => $order->payment_status,
            'payment_note' => null,
            'payment_sender' => null,
            'payment_trxid' => $order->payment_trxid,
            'shipping' => $order->getShippingAddress(),
            'cart_items' => $order->items->map(function($item) {
                return [
                    'id' => $item->product_id,
                    'variant_id' => $item->product_variation_id,
                    'title' => $item->product->title,
                    'image' => $item->product->primary_image?->url ?? asset('assets/default_img.png'),
                    'price' => $item->getUnitPrice(),
                    'quantity' => $item->quantity,
                    'unit' => $item->variation->unit_value . ' ' . $item->variation->unit_type,
                    'total' => $item->getTotal()
                ];
            })->toArray(),
            'totals' => $order->getTotals(),
            'coupon' => $order->getCouponDetails(),
            'show_success' => session()->has('order_success'),
            'payment_failed' => session()->has('payment_failed'),
            'order_status' => $order->order_status,
            'is_in_progress' => $order->isInProgress()
        ];

        return view('product.order_invoice', compact('orderData'));
    }

    public function retryPayment(Request $request, $order_id)
    {
        $order = Order::where('order_number', $order_id)->firstOrFail();
        if ($order->payment_method !== 'sslcommerz') {
            return redirect()->route('product.order.invoice', ['order_id' => $order->order_number]);
        }

        $postData = [
            'store_id' => config('services.sslcommerz.store_id'),
            'store_passwd' => config('services.sslcommerz.store_password'),
            'total_amount' => $order->total,
            'currency' => 'BDT',
            'tran_id' => $order->order_number,
            'success_url' => route('payment.ssl.success'),
            'fail_url' => route('payment.ssl.fail'),
            'cancel_url' => route('payment.ssl.cancel'),
            'ipn_url' => route('payment.ssl.ipn'),
            'cus_name' => $order->contact_name,
            'cus_email' => $order->contact_email ?? 'noemail@example.com',
            'cus_add1' => $order->address,
            'cus_city' => $order->city,
            'cus_state' => $order->state,
            'cus_postcode' => $order->postal_code,
            'cus_country' => 'Bangladesh',
            'cus_phone' => $order->contact_phone,
            'shipping_method' => 'NO',
            'product_name' => 'Order ' . $order->order_number,
            'product_category' => 'Mixed',
            'product_profile' => 'general'
        ];

        $gatewayUrl = (bool) config('services.sslcommerz.sandbox', true)
            ? 'https://sandbox.sslcommerz.com/gwprocess/v4/api.php'
            : 'https://securepay.sslcommerz.com/gwprocess/v4/api.php';

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $gatewayUrl);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postData));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $gw = json_decode($response, true);
        if ($httpCode === 200 && isset($gw['status']) && $gw['status'] === 'SUCCESS' && !empty($gw['GatewayPageURL'])) {
            return redirect()->away($gw['GatewayPageURL']);
        }

        return redirect()->route('product.order.invoice', ['order_id' => $order->order_number])
            ->with('error', 'Failed to initiate payment retry. Please try again later.');
    }

    /**
     * Get cart items data for the cart sidebar
     */
    public function getCartItems(Request $request)
    {
        // Get cart data from request body
        $cartItems = json_decode($request->getContent(), true) ?? [];
        
        if (empty($cartItems)) {
            return response()->json([
                'items' => [],
                'totals' => [
                    'sub_total' => 0,
                    'total' => 0
                ]
            ]);
        }

        // Fetch products with their variations and primary images
        $products = Product::with(['variations', 'images' => function($q) {
            $q->where('is_primary', true);
        }])->whereIn('id', array_column($cartItems, 'productId'))->get();

        $items = [];
        $subTotal = 0;

        foreach ($cartItems as $cartItem) {
            $product = $products->find($cartItem['productId']);
            if (!$product) continue;

            $variation = $product->variations->find($cartItem['variantId']);
            if (!$variation) continue;

            $itemTotal = $variation->sale_price * $cartItem['quantity'];
            $subTotal += $itemTotal;

            $items[] = [
                'id' => $product->id,
                'variant_id' => $variation->id,
                'title' => $product->title,
                'image' => $product->primary_image?->url ?? asset('assets/default_img.png'),
                'price' => $variation->sale_price,
                'quantity' => $cartItem['quantity'],
                'stock' => $variation->in_stock,
                'total' => $itemTotal,
                'unit' => $variation->unit_value . ' ' . $variation->unit_type,
                'url' => route('product.view', $product->slug)
            ];
        }

        return response()->json([
            'items' => $items,
            'totals' => [
                'sub_total' => $subTotal,
                'total' => $subTotal
            ]
        ]);
    }

    // New: Order Tracking page (no logic yet)
    public function orderTrack(Request $request)
    {
        // If no query parameters, just show the form
        if (!$request->filled(['email', 'order_id'])) {
            return view('product.order_track');
        }

        // Validate input
        $request->validate([
            'email' => 'required|email',
            'order_id' => 'required|string'
        ]);

        // Find order with matching order number and either contact_email or user's email
        $order = Order::with(['statuses' => function($q) {
                    $q->orderBy('created_at', 'asc');
                }])
                ->where('order_number', $request->order_id)
                ->where(function($query) use ($request) {
                    $query->where('contact_email', $request->email)
                          ->orWhereHas('user', function($q) use ($request) {
                              $q->where('email', $request->email);
                          });
                })
                ->first();

        if (!$order) {
            return redirect()
                ->route('product.order.track')
                ->with('error', 'No order found with the provided details. Please check your email and order ID.');
        }

        // Get status history with timestamps
        $statusHistory = $order->statuses->map(function($status) {
            return [
                'status' => $status->status,
                'timestamp' => $status->created_at->format('M d, Y h:i A')
            ];
        });

        // Get current statuses for the progress bar
        $statusInfo = [
            'current' => $order->getCurrentStatus(),
            'history' => $statusHistory,
            'statuses' => [
                'pending' => $order->getCurrentStatus() === Order::STATUS_PENDING,
                'processing' => $order->getCurrentStatus() === Order::STATUS_PROCESSING,
                'shipped' => $order->getCurrentStatus() === Order::STATUS_SHIPPED,
                'delivered' => $order->getCurrentStatus() === Order::STATUS_DELIVERED,
                'cancelled' => $order->getCurrentStatus() === Order::STATUS_CANCELLED
            ]
        ];

        return view('product.order_track', compact('statusInfo'));
    }

    public function search(Request $request)
    {
        try {
            $query = $request->get('query');
            
            if (empty($query)) {
                // Return latest products as default suggestions
                $products = Product::with(['variations' => function($q) {
                        $q->where('default', true)
                          ->select('id', 'product_id', 'sale_price', 'default');
                    }])
                    ->with(['images' => function($q) {
                        $q->where('is_primary', true)
                          ->select('id', 'product_id', 'path', 'is_primary');
                    }])
                    ->select('id', 'title', 'slug')
                    ->latest()
                    ->take(5)
                    ->get()
                    ->map(function($product) {
                        $defaultVariation = $product->variations->first();
                        $primaryImage = $product->images->first();
                        
                        return [
                            'id' => $product->slug, // Using slug for the URL
                            'name' => $product->title,
                            'thumbnail' => $primaryImage ? asset('storage/' . $primaryImage->path) : asset('assets/default_img.png'),
                            'default_price' => $defaultVariation ? $defaultVariation->sale_price : 0
                        ];
                    });

                return response()->json($products);
            }

            $products = Product::where('title', 'like', "%{$query}%")
                ->with(['variations' => function($q) {
                    $q->where('default', true)
                      ->select('id', 'product_id', 'sale_price', 'default');
                }])
                ->with(['images' => function($q) {
                    $q->where('is_primary', true)
                      ->select('id', 'product_id', 'path', 'is_primary');
                }])
                ->select('id', 'title', 'slug')
                ->take(5)
                ->get()
                ->map(function($product) {
                    $defaultVariation = $product->variations->first();
                    $primaryImage = $product->images->first();
                    
                    return [
                        'id' => $product->slug, // Using slug for the URL
                        'name' => $product->title,
                        'thumbnail' => $primaryImage ? asset('storage/' . $primaryImage->path) : asset('assets/default_img.png'),
                        'default_price' => $defaultVariation ? $defaultVariation->sale_price : 0
                    ];
                });

            return response()->json($products);
        } catch (\Exception $e) {
            Log::error('Product search error: ' . $e->getMessage());
            return response()->json(['error' => 'An error occurred while searching'], 500);
        }
    }
}