<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Auth\Events\PasswordReset;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Str;
use Illuminate\View\View;

class NewPasswordController extends Controller
{
    public function create(Request $request): View
    {
        return view('auth.reset_password', ['request' => $request]);
    }

    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'token' => ['required'],
            'email' => ['required', 'email'],
            'password' => [
                'required',
                'confirmed',
                'min:6',
                'max:128'
            ],
            'g-recaptcha-response' => ['required', 'captcha']
        ], [
            'password.min' => 'The password must be at least 6 characters.',
            'password.max' => 'The password cannot be longer than 128 characters.',
            'g-recaptcha-response.required' => 'The captcha field is required.'
        ]);

        $errors = [];

        $status = Password::reset(
            $request->only('email', 'password', 'password_confirmation', 'token'),
            function ($user) use ($request) {
                $user->forceFill([
                    'password' => Hash::make($request->password),
                    'remember_token' => Str::random(60),
                ])->save();

                event(new PasswordReset($user));
            }
        );

        if ($status != Password::PASSWORD_RESET) {
            if ($status === Password::INVALID_TOKEN) {
                $errors['token'] = 'This password reset link has expired or is invalid.';
                $errors['suggestion'] = 'Please request a new password reset link.';
            } else if ($status === Password::INVALID_USER) {
                $errors['email'] = 'We could not find a user with that email address.';
            } else {
                $errors['error'] = 'Unable to reset password.';
                $errors['help'] = 'Please try again or contact support if the problem persists.';
            }
            return back()
                ->withInput($request->only('email'))
                ->withErrors($errors);
        }

        return redirect()->route('login')
            ->with('success', 'Your password has been reset successfully! You can now login with your new password.');
    }
}