<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Resources\ProductResource;
use App\Models\Product;
use App\Models\Setting;
use Illuminate\Http\Request;

class ProductExportController extends Controller
{
    public function index(Request $request)
    {
        $settings = Setting::singleton();
        $toggles = $settings->api_data_toggles ?? [];

        $with = [];
        if (($toggles['category'] ?? true)) { $with[] = 'category'; }
        if (($toggles['brand'] ?? true)) { $with[] = 'brand'; }
        if (($toggles['images'] ?? true)) { $with[] = 'images'; }
        if (($toggles['stocks'] ?? true)) { $with[] = 'variations'; }

        $query = Product::query()->with($with);

        if ($search = $request->query('q')) {
            $query->where('title', 'like', "%{$search}%");
        }

        // Optional: filter by category slug or brand title
        if ($category = $request->query('category')) {
            $query->whereHas('category', function ($q) use ($category) {
                $q->where('slug', $category)->orWhere('name', $category);
            });
        }

        if ($brand = $request->query('brand')) {
            $query->whereHas('brand', function ($q) use ($brand) {
                $q->where('title', $brand);
            });
        }

        if (!($toggles['products'] ?? true)) {
            return response()->json(['message' => 'Products export is disabled'], 403);
        }

        $perPage = min((int) $request->query('per_page', 50), 200);
        $products = $query->paginate($perPage);

        $collection = $products->getCollection()->map(function ($product) use ($toggles, $request) {
            return (new ProductResource($product))->withToggles($toggles)->toArray($request);
        });

        return response()->json([
            'products' => $collection,
            'meta' => [
                'current_page' => $products->currentPage(),
                'per_page' => $products->perPage(),
                'total' => $products->total(),
                'last_page' => $products->lastPage(),
            ],
        ]);
    }

    public function all(Request $request)
    {
        $settings = Setting::singleton();
        $toggles = $settings->api_data_toggles ?? [];

        if (!($toggles['products'] ?? true)) {
            return response()->json(['message' => 'Products export is disabled'], 403);
        }

        $with = [];
        if (($toggles['category'] ?? true)) { $with[] = 'category'; }
        if (($toggles['brand'] ?? true)) { $with[] = 'brand'; }
        if (($toggles['images'] ?? true)) { $with[] = 'images'; }
        if (($toggles['stocks'] ?? true)) { $with[] = 'variations'; }

        $products = Product::with($with)->orderByDesc('created_at')->get();

        $collection = $products->map(function ($product) use ($toggles, $request) {
            return (new ProductResource($product))->withToggles($toggles)->toArray($request);
        });

        return response()->json(['data' => $collection, 'meta' => ['total' => $products->count()]]);
    }

    public function show(Product $product, Request $request)
    {
        $settings = Setting::singleton();
        $toggles = $settings->api_data_toggles ?? [];

        $product->load(['category', 'brand', 'images', 'variations']);

        return response()->json([
            'product' => (new ProductResource($product))->withToggles($toggles)->toArray($request)
        ]);
    }

    public function showByTitle(string $title, Request $request)
    {
        $product = Product::where('title', $title)->firstOrFail();
        return $this->show($product, $request);
    }

    public function latest(int $limit, Request $request)
    {
        $settings = Setting::singleton();
        $toggles = $settings->api_data_toggles ?? [];

        if (!($toggles['products'] ?? true)) {
            return response()->json(['message' => 'Products export is disabled'], 403);
        }

        $limit = max(1, min($limit, 200));

        $with = [];
        if (($toggles['category'] ?? true)) { $with[] = 'category'; }
        if (($toggles['brand'] ?? true)) { $with[] = 'brand'; }
        if (($toggles['images'] ?? true)) { $with[] = 'images'; }
        if (($toggles['stocks'] ?? true)) { $with[] = 'variations'; }

        $products = Product::with($with)
            ->orderByDesc('created_at')
            ->take($limit)
            ->get();

        $collection = $products->map(function ($product) use ($toggles, $request) {
            return (new ProductResource($product))->withToggles($toggles)->toArray($request);
        });

        return response()->json(['products' => $collection, 'meta' => ['limit' => $limit]]);
    }
}


