<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use App\Models\Setting;
use App\Services\EnvUpdater;
use Illuminate\Support\Str;

class SettingsController extends Controller
{
    public function index()
    {
        $settings = Setting::singleton();

        $env = [
            'APP_NAME' => config('app.name'),
            'APP_URL' => config('app.url'),
            'MAIL_MAILER' => config('mail.default'),
            'MAIL_HOST' => config('mail.mailers.smtp.host'),
            'MAIL_PORT' => config('mail.mailers.smtp.port'),
            'MAIL_USERNAME' => config('mail.mailers.smtp.username'),
            'MAIL_PASSWORD' => config('mail.mailers.smtp.password'),
            'MAIL_ENCRYPTION' => config('mail.mailers.smtp.encryption'),
            'MAIL_FROM_ADDRESS' => config('mail.from.address'),
            'MAIL_FROM_NAME' => config('mail.from.name'),
            'ADMIN_NOTIFIER_MAIL' => config('mail.admin_notifier_mail'),
            'NOCAPTCHA_SITEKEY' => config('captcha.sitekey'),
            'NOCAPTCHA_SECRET' => config('captcha.secret'),
            'GOOGLE_CLIENT_ID' => config('services.google.client_id'),
            'GOOGLE_CLIENT_SECRET' => config('services.google.client_secret'),
            'GOOGLE_REDIRECT_URI' => config('services.google.redirect'),
            'TAWK_KEY' => config('services.tawk.key'),
            'SSLCZ_STORE_ID' => config('services.sslcommerz.store_id'),
            'SSLCZ_STORE_PASSWORD' => config('services.sslcommerz.store_password'),
            // Cast to explicit int so the view can reliably compare 0/1
            'SSLCZ_SANDBOX' => (int) (config('services.sslcommerz.sandbox') ? 1 : 0)
        ];

        return view('admin.settings.index', compact('settings', 'env'));
    }

    public function update(Request $request)
    {
        $settings = Setting::singleton();

        $validated = $request->validate([
            'site_logo' => 'nullable|image|mimes:png,jpg,jpeg,svg,webp|max:2048',
            'admin_logo' => 'nullable|image|mimes:png,jpg,jpeg,svg,webp|max:2048',
            'favicon' => 'nullable|image|mimes:png,jpg,jpeg,ico|max:1024',
            'footer_copyright' => 'nullable|string|max:255',
            'google_analytics_id' => 'nullable|string|max:64',
            'meta_keywords' => 'nullable|string',
            'meta_description' => 'nullable|string|max:500',
            'loader_image' => 'nullable|image|mimes:png,jpg,jpeg,svg,webp|max:2048',
            'pixel_code' => 'nullable|string',
            'contact_address' => 'nullable|string|max:255',
            'contact_phone' => 'nullable|string|max:50',
            'contact_email' => 'nullable|email|max:255',
            'social_media' => 'nullable|array',
            'social_media.*.platform' => 'required|string|max:50',
            'social_media.*.url' => 'required|url|max:255',
            'footer_description' => 'nullable|string|max:500',
            // API fields
            'api_enabled' => 'sometimes|boolean',
            'api_data_toggles' => 'sometimes|array',
            'api_data_toggles.products' => 'sometimes|boolean',
            'api_data_toggles.category' => 'sometimes|boolean',
            'api_data_toggles.brand' => 'sometimes|boolean',
            'api_data_toggles.images' => 'sometimes|boolean',
            'api_data_toggles.stocks' => 'sometimes|boolean',
            'api_data_toggles.ratings' => 'sometimes|boolean',
            'regenerate_api_key' => 'sometimes|boolean'
        ]);

        if ($request->hasFile('site_logo')) {
            if ($settings->site_logo_path) {
                Storage::disk('public')->delete($settings->site_logo_path);
            }
            $validated['site_logo_path'] = $request->file('site_logo')->store('site-assets', 'public');
        }

        if ($request->hasFile('admin_logo')) {
            if ($settings->admin_logo_path) {
                Storage::disk('public')->delete($settings->admin_logo_path);
            }
            $validated['admin_logo_path'] = $request->file('admin_logo')->store('site-assets', 'public');
        }

        if ($request->hasFile('favicon')) {
            if ($settings->favicon_path) {
                Storage::disk('public')->delete($settings->favicon_path);
            }
            $validated['favicon_path'] = $request->file('favicon')->store('site-assets', 'public');
        }

        if ($request->hasFile('loader_image')) {
            if ($settings->loader_image_path) {
                Storage::disk('public')->delete($settings->loader_image_path);
            }
            $validated['loader_image_path'] = $request->file('loader_image')->store('site-assets', 'public');
        }

        if ($request->filled('pixel_code')) {
            $validated['pixel_code'] = $request->input('pixel_code');
        } else {
            $validated['pixel_code'] = null;
        }

        // Handle social media data
        if ($request->filled('social_media')) {
            $socialMedia = [];
            foreach ($request->input('social_media') as $item) {
                if (!empty($item['platform']) && !empty($item['url'])) {
                    $socialMedia[] = [
                        'platform' => $item['platform'],
                        'url' => $item['url']
                    ];
                }
            }
            $validated['social_media'] = $socialMedia;
        } else {
            $validated['social_media'] = [];
        }

        // Normalize API toggles (fill defaults if missing)
        $toggles = $settings->api_data_toggles ?? [];
        $newToggles = array_merge([
            'products' => true,
            'category' => true,
            'brand' => true,
            'images' => true,
            'stocks' => true,
            'ratings' => true,
        ], (array)($request->input('api_data_toggles') ?? []));

        $validated['api_data_toggles'] = $newToggles;

        // Handle API enable/disable
        if ($request->has('api_enabled')) {
            $validated['api_enabled'] = (bool) $request->boolean('api_enabled');
        }

        // Regenerate API key if requested or missing
        $shouldRegenerate = $request->boolean('regenerate_api_key') || empty($settings->api_key);
        if ($shouldRegenerate) {
            $validated['api_key'] = Str::random(32);
        }

        $settings->update($validated);

        // Env updates section
        $envInput = $request->only([
            'APP_NAME','APP_URL','MAIL_MAILER','MAIL_HOST','MAIL_PORT','MAIL_USERNAME','MAIL_PASSWORD','MAIL_ENCRYPTION','MAIL_FROM_ADDRESS','MAIL_FROM_NAME','ADMIN_NOTIFIER_MAIL','NOCAPTCHA_SITEKEY','NOCAPTCHA_SECRET','GOOGLE_CLIENT_ID','GOOGLE_CLIENT_SECRET','GOOGLE_REDIRECT_URI','TAWK_KEY','SSLCZ_STORE_ID','SSLCZ_STORE_PASSWORD','SSLCZ_SANDBOX'
        ]);

        if (!empty($envInput)) {
            EnvUpdater::bulkSet($envInput);
            // Clear config cache to reflect .env changes
            \Artisan::call('config:clear');
        }

        return back()->with('success', 'Settings updated successfully.');
    }
} 