<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\OrderItem;
use App\Models\Order;
use App\Models\ProductRating;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ProductAnalyticsController extends Controller
{
    protected $completedStatuses = ['completed', 'delivered'];

    public function show(Product $product)
    {
        $currentYear = Carbon::now()->year;

        // Get all months of current year
        $months = collect();
        for ($month = 1; $month <= 12; $month++) {
            $months->push(Carbon::createFromDate($currentYear, $month, 1)->format('Y-m'));
        }

        // Get sales data for current year
        $salesData = OrderItem::query()
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->join('product_variations', 'order_items.product_variation_id', '=', 'product_variations.id')
            ->where('order_items.product_id', $product->id)
            ->whereIn('orders.order_status', $this->completedStatuses)
            ->whereYear('orders.created_at', $currentYear)
            ->select(
                DB::raw('DATE_FORMAT(orders.created_at, "%Y-%m") as month'),
                DB::raw('SUM(order_items.quantity) as total_quantity'),
                DB::raw('SUM(product_variations.sale_price * order_items.quantity) as total_revenue')
            )
            ->groupBy(DB::raw('DATE_FORMAT(orders.created_at, "%Y-%m")'))
            ->get()
            ->keyBy('month');

        // Fill in missing months with zero values
        $monthlySales = $months->map(function($month) use ($salesData) {
            return $salesData->get($month, [
                'month' => $month,
                'total_quantity' => 0,
                'total_revenue' => 0
            ]);
        })->values();

        // Get variant sales data with proper formatting
        $variantSales = OrderItem::query()
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->join('product_variations', 'order_items.product_variation_id', '=', 'product_variations.id')
            ->where('order_items.product_id', $product->id)
            ->whereIn('orders.order_status', $this->completedStatuses)
            ->select(
                'order_items.product_variation_id',
                'product_variations.unit_value',
                'product_variations.unit_type',
                DB::raw('SUM(order_items.quantity) as total_quantity'),
                DB::raw('SUM(product_variations.sale_price * order_items.quantity) as total_revenue')
            )
            ->groupBy(
                'order_items.product_variation_id',
                'product_variations.unit_value',
                'product_variations.unit_type'
            )
            ->orderBy('total_quantity', 'desc')
            ->get()
            ->map(function ($variant) {
                $variant->label = $variant->unit_value . 'GB/' . $variant->unit_type;
                return $variant;
            });

        // Get rating distribution
        $ratingDistribution = $product->ratings()
            ->select('rating', DB::raw('COUNT(*) as count'))
            ->groupBy('rating')
            ->orderBy('rating', 'desc')
            ->get();

        // Calculate total sales
        $totalSales = OrderItem::query()
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->where('order_items.product_id', $product->id)
            ->whereIn('orders.order_status', $this->completedStatuses)
            ->sum('order_items.quantity');

        // Calculate total revenue
        $totalRevenue = OrderItem::query()
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->join('product_variations', 'order_items.product_variation_id', '=', 'product_variations.id')
            ->where('order_items.product_id', $product->id)
            ->whereIn('orders.order_status', $this->completedStatuses)
            ->sum(DB::raw('product_variations.sale_price * order_items.quantity'));

        // Get this month's sales
        $thisMonthSales = OrderItem::query()
            ->join('orders', 'order_items.order_id', '=', 'orders.id')
            ->where('order_items.product_id', $product->id)
            ->whereIn('orders.order_status', $this->completedStatuses)
            ->whereMonth('orders.created_at', Carbon::now()->month)
            ->whereYear('orders.created_at', Carbon::now()->year)
            ->sum('order_items.quantity');

        // Get recent reviews with pagination
        $recentReviews = $product->ratings()
            ->with('user')
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('admin.products.analytics', compact(
            'product',
            'monthlySales',
            'variantSales',
            'ratingDistribution',
            'totalSales',
            'totalRevenue',
            'thisMonthSales',
            'recentReviews'
        ));
    }

    public function deleteReview($id)
    {
        $review = ProductRating::findOrFail($id);
        $review->delete();
        return redirect()->back()->with('success', 'Review deleted successfully');
    }
} 