<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\BestDeals;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class BestDealsController extends Controller
{
    /**
     * Display a listing of the deals.
     */
    public function index()
    {
        $deals = BestDeals::with('products')->latest()->paginate(10);
        return view('admin.products.deals.list', compact('deals'));
    }

    /**
     * Show the form for creating a new deal.
     */
    public function create()
    {
        $products = Product::orderBy('title')->get();
        return view('admin.products.deals.create', compact('products'));
    }

    /**
     * Store a newly created deal in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'products' => 'required|array',
            'products.*' => 'exists:products,id'
        ]);

        try {
            DB::beginTransaction();

            $deal = BestDeals::create([
                'name' => $request->name,
                'description' => $request->description,
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
                'status' => $request->boolean('status', true)
            ]);

            $deal->products()->attach($request->products);

            DB::commit();
            return redirect()->route('admin.deals')
                           ->with('success', 'Deal created successfully. ' . 
                                ($deal->status ? 'This deal is now active and all other deals have been deactivated.' : ''));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to create deal. Please try again.');
        }
    }

    /**
     * Show the form for editing the specified deal.
     */
    public function edit(BestDeals $deal)
    {
        $products = Product::orderBy('title')->get();
        $selectedProducts = $deal->products->pluck('id')->toArray();
        return view('admin.products.deals.edit', compact('deal', 'products', 'selectedProducts'));
    }

    /**
     * Update the specified deal in storage.
     */
    public function update(Request $request, BestDeals $deal)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'products' => 'required|array',
            'products.*' => 'exists:products,id'
        ]);

        try {
            DB::beginTransaction();

            $wasInactive = !$deal->status;
            $willBeActive = $request->boolean('status', true);

            $deal->update([
                'name' => $request->name,
                'description' => $request->description,
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
                'status' => $willBeActive
            ]);

            $deal->products()->sync($request->products);

            DB::commit();

            $message = 'Deal updated successfully.';
            if ($wasInactive && $willBeActive) {
                $message .= ' This deal is now active and all other deals have been deactivated.';
            }

            return redirect()->route('admin.deals')
                           ->with('success', $message);
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to update deal. Please try again.');
        }
    }

    /**
     * Remove the specified deal from storage.
     */
    public function destroy(BestDeals $deal)
    {
        try {
            $deal->delete();
            return redirect()->route('admin.deals')
                           ->with('success', 'Deal deleted successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to delete deal. Please try again.');
        }
    }
} 