<?php

namespace App\Exports;

use App\Models\Order;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;

class OrdersExport implements FromCollection, WithHeadings, WithMapping, WithColumnWidths, WithStyles
{
    public function collection()
    {
        return Order::with(['user', 'items.product', 'items.variation', 'items.product.category', 'payment.paymentMethod'])
            ->latest()
            ->get();
    }

    public function headings(): array
    {
        return [
            'Order ID',
            'Order Date',
            'Status',
            'Customer Name',
            'Customer Email',
            'Customer Phone',
            'Shipping Address',
            'Products',
            'Categories',
            'Total Amount',
            'Payment Method',
            'Payment Status'
        ];
    }

    public function map($order): array
    {
        // Format products as a clean list
        $products = collect($order->items)->map(function($item) {
            $name = $item->product->title;
            if ($item->variation) {
                $name .= ' (' . $item->variation->name . ')';
            }
            return $name . ' () x ' . $item->quantity;
        })->implode("\n");

        // Format categories
        $categories = collect($order->items)
            ->pluck('product.category.name')
            ->unique()
            ->implode("\n");

        // Format phone number to be treated as text
        $phone = $order->contact_phone ? "'" . $order->contact_phone : 'N/A';

        return [
            $order->order_number,
            $order->created_at->format('Y-m-d H:i:s'),
            $order->order_status ?: 'N/A',
            $order->contact_name ?: 'N/A',
            $order->user->email ?: 'N/A',
            $phone,
            $order->shipping_address ? ($order->shipping_address . ', ' . $order->shipping_city) : 'N/A',
            $products ?: 'N/A',
            $categories ?: 'N/A',
            number_format($order->total, 2) . ' N/A',
            $order->payment && $order->payment->paymentMethod ? $order->payment->paymentMethod->name : 'N/A',
            $order->payment ? $order->payment->status : 'N/A'
        ];
    }

    public function columnWidths(): array
    {
        return [
            'A' => 20,  // Order ID
            'B' => 20,  // Date
            'C' => 15,  // Status
            'D' => 20,  // Customer Name
            'E' => 30,  // Email
            'F' => 15,  // Phone
            'G' => 30,  // Address
            'H' => 40,  // Products
            'I' => 15,  // Categories
            'J' => 15,  // Total
            'K' => 15,  // Payment Method
            'L' => 15,  // Payment Status
        ];
    }

    public function styles(Worksheet $sheet)
    {
        // Get the highest row number
        $highestRow = $sheet->getHighestRow();
        
        return [
            // Header styling
            1 => [
                'font' => ['bold' => true],
                'fill' => ['fillType' => 'solid', 'startColor' => ['rgb' => 'E9ECEF']],
                'alignment' => ['horizontal' => Alignment::HORIZONTAL_LEFT]
            ],
            // Style for all data cells
            'A2:L' . $highestRow => [
                'alignment' => ['horizontal' => Alignment::HORIZONTAL_LEFT]
            ],
        ];
    }
} 